function [logP,optQ] = log_cont_viterbi_alg(a,mu,sigma,c,X);
%
%  function [logP,optQ] = log_cont_viterbi_alg(a,mu,sigma,c,X);
%
%  This implementation of the Viterbi algorithm evaluates in the 
%  logarithmic domain for the given observation sequence `X` and the 
%  continuous density HMM (given by `a`, `mu`, `sigma` and `c`) the 
%  optimal state sequence `optQ`, i.e. the state sequence with the 
%  highest joint probability P(Q,X|HMM). If the HMM is not able to 
%  generate the given observation sequence, the probability is zero 
%  and the optimal state sequence is empty.
%
%  Inputs:  a      state transition probabilities (NxN matrix),
%                  where a(i,j) denotes the probability to move from 
%                  state i to state j
%           mu     mean values of observation densities (NxMxD matrix) 
%           sigma  variances of observation densities (NxMxD matrix) 
%                  (the covariance matrix is assumed to be diagonal) 
%           c      mixture weigths (NxM matrix) 
%           X      observation sequence (TxD matrix)
%
%                  The dimensions of the above arguments denote:
%                    N:  number of HMM states incl. start and end state
%                    M:  number of mixture components
%                    D:  dimension of feature vectors
%                    T:  length of observation sequence
%
%  Output:  logP   log joint probability of the observation sequence and 
%                  the optimal state sequence given the continuous 
%                  density HMM {a,mu,sigma,c}
%           optQ   optimal state sequence, i.e. state sequence with
%                  the highest likelihood (length of optQ is T+2, 
%                  whereby optQ(1) = 1 and optQ(T+2) = N)
%
%
%  All calculations are done in log space to prevent underflow.

if nargin < 5,
  error('missing input arguments');
end;

N = size(a,1);   % number of states 
M = size(c,2);   % number of mixtures
D = size(X,2);   % dimension of the observations
T = size(X,1);   % length of observation sequence

if size(a,2) ~= N | size(mu,1) ~= N | size(mu,2) ~= M | ...
   size(mu,3) ~= D | size(sigma,1) ~= N | size(sigma,2) ~= M | ...
   size(sigma,3) ~= D | size(c,1) ~= N 
  error(['*** dimensions of arguments `a`, `mu`, `sigma` and ' ...
         '`c` do not agree'])
end
if length(X) == 0
  error('*** empty observation sequence is not allowed')
end

logA = logProb(a);   % logarithm of transition probabilities
logC = logProb(c);   % logarithm of mixture weights

                     % generate log likelihoods of all states and observations 
logB = -Inf * ones(N,T);
sqsigma = sqrt(sigma);    % (standard deviation = sqrt of variance)
for j = 2:N-1
  mu_j = shiftdim(mu(j,:,:),1);             % put mu and sigma of state j
  sqsigma_j = shiftdim(sqsigma(j,:,:),1);   % in 2-dimensional matrix 
  for t = 1:T
    for k = 1:M
      aux = logC(j,k)+sum(logProb(normpdf(X(t,:),mu_j(k,:),sqsigma_j(k,:))));
      logB(j,t) = logSum(logB(j,t),aux);
    end
  end
end

%----- Viterbi algorithm -----

logDelta = -Inf * ones(T,N);
psi = zeros(T,N);

for j = 2:N-1                                    % initialization
  logDelta(1,j) = logA(1,j)+logB(j,1);
end

for t = 2:T                                      % recursion
  for j = 2:N-1
    mx = -Inf;
    argMax = 0;
    for i = 2:N-1
      if logDelta(t-1,i)+logA(i,j) >= mx 
        mx = logDelta(t-1,i)+logA(i,j);
        argMax = i;
      end
    end
    logDelta(t,j) = mx+logB(j,t);
    psi(t,j) = argMax;
  end
end

mx = -Inf;                                       % termination
argMax = 0;
for i = 2:N-1
  if logDelta(T,i)+logA(i,N) >= mx 
    mx = logDelta(T,i)+logA(i,N);
    argMax = i;
  end
end
logDelta(T,N) = mx;
psi(T,N) = argMax;

logP = logDelta(T,N);
if logP == -Inf                    % return empty optQ if probability = 0
  optQ = [];
  return
end

optQ = zeros(1,T);                 % backward search of optimal path
optQ(T) = psi(T,N);
for t = T-1:-1:1
  optQ(t) = psi(t+1,optQ(t+1));
end
optQ = [1 optQ N];

