% Uebung 22
%
% Einzelwort-Erkennung mit CDHMM
% -------------------------------------------------------
%
% Uebung 22.2 
%
% Erkennen der Ziffern in den Testsignalen
% ----------------------------------------
%
% Mit dieser Matlab-Funktion koennen die im File cdhmms.mat vorhandenen
% CDHMM eingesetzt werden, um gesprochene Woerter zu erkennen. Die
% Funktion wird wie folgt eingesetzt: 
%
%   ueb22_2(filnam,alg)
%   
% Dabei ist `filnam` eine optionale Spezifikation der Sprachsignaldateien, 
% fuer welche die Spracherkennung ausgefuehrt werden soll; Wildcards sind 
% erlaubt. Der vorgegebene Name ist: '*test.wav', was das ganze Test-Set 
% bedeutet. Die Struktur der Dateinamen ist  "zXX_rY_sZZ_test.wav",  wobei
%    XX: Wortnummer von 01 bis 11 ("eins", "zwei", ... , "null", "zwo")
%    Y:  Repetitionsnummer von 1 bis 5
%    ZZ: Sprechernummer von 01 bis 12
%
% Mit dem zweiten optionalen Argument `alg` kann der Forward- oder der 
% Viterbi-Algorithmus ausgewaehlt werden mit 'fw' bzw. 'vt' (default).  
%
% Die Resultate der Worterkennung werden in der Datei 'results.mat' 
% gespeichert und zwar als Struktur:
%
%    result(i).file           Name der Signaldatei 
%    result(i).speaker        Sprechernummer 
%    result(i).is_word_num    Nummer des gesprochenen Wortes
%    result(i).rec_word_num   Nummer des erkannten Wortes


function ueb22_2(filnam,alg)

if nargin < 1 | isempty(filnam) 
  filnam = '*test.wav';
else
  if strcmp(filnam(end-3:end),'.wav') == 0
    error(['*** Argument ist ungueltig:  ' filnam]) 
  end
end;
if nargin < 2
  alg = 'vt';
end

starttime = clock;


%----- trainierte CDHMM laden -----
%
% cdhmm(i).name    Name (modelliertes Wort)
% cdhmm(i).a       Uebergangswahrscheinlichkeiten (NxN-Matrix) 
% cdhmm(i).mu      Mittelwerte (NxMxD-Matrix)
% cdhmm(i).sigma   Varianzen (NxMxD-Matrix)
% cdhmm(i).c       Mischgewichte (NxM-Matrix)
%
% N = Anzahl Zustaende
% D = Dimension der Beobachtungen
% M = Anzahl Mischkomponenten

if exist('cdhmms.mat') ~= 2,
  disp(['Die Datei ''cdhmms.mat'' mit den trainierten CDHMM wurde ', ...
       'nicht gefunden.']);
  return;
end;
load cdhmms.mat;


%----- Namen der Testsignal-Files bestimmen -----

sigdirnam = fullpath('ueb22_sigs/');
filfound = dir([sigdirnam filnam]);
nfils = length(filfound);
disp(['Anzahl der gefundenen Dateien: ',num2str(nfils)]);
if length(filfound) == 0
  return  
end


%----- Erkennung -----

result = [];
if strcmp(alg,'fw')
  disp('Erkennung mit Forward-Algorithmus ...')
elseif strcmp(alg,'vt')
  disp('Erkennung mit Viterbi-Algorithmus ...')
else
  error(['*** ungueltiger Algorithmus:  ''' alg ''''])
end    
ncorr = 0;
disp('Signal   gespr. Wort   erk. Wort')
for i = 1:nfils
  feafilnam = [filfound(i).name(1:end-4) '_cep.mat'];
  feadirnam = 'ueb22_data/';
  if exist([feadirnam feafilnam],'file')    % laden oder berechnen der MFCC 
    load([feadirnam feafilnam]);
  else
    % disp(['Merkmal-File nicht vorhanden:  ' feafilnam])
    % disp(['(Merkmale werden berechnet)'])
    [sig,fs] = audioread([sigdirnam filfound(i).name]); 
    Ncep = 12;         % Anzahl cepstrale Coeffizienten 
    Nfilt = 24;        % Groesse der Mel-Filterbank
    WinSize = 300;     % Laenge des Analysefensters
    WinShift = 100;    % Verschiebung des Analysefensters
    win = hamming(WinSize);
    cep = extr_mel_ceps(sig,fs,win,WinShift,Nfilt,Ncep);
  end
  
  prob = [];                                 % erkennen mit allen CDHMM
  for k = 1:size(cdhmm,2),
    if strcmp(alg,'fw')
      prob(k) = log_cont_forward_alg(cdhmm(k).a,cdhmm(k).mu, ...
                                     cdhmm(k).sigma,cdhmm(k).c,cep);
    else
      [prob(k),optQ] = log_cont_viterbi_alg(cdhmm(k).a,cdhmm(k).mu, ...
                                     cdhmm(k).sigma,cdhmm(k).c,cep);
    end
  end

  [maxVal,maxInd] = max(prob);               % Resultat eintragen
  result(i).file = filfound(i).name;
  sprZiffer = findstr(feafilnam,'_s')+2;
  result(i).speaker = str2num(feafilnam(sprZiffer:sprZiffer+1));
  result(i).is_word_num = str2num(filfound(i).name(2:3));
  result(i).rec_word_num = maxInd;
  if result(i).is_word_num == result(i).rec_word_num
    ncorr = ncorr+1;
  end
  fprintf(1,'  %3d ',i);
  fprintf(1,'%s ',['      ' cdhmm(result(i).is_word_num).name ...
                   '      ' cdhmm(result(i).rec_word_num).name])
  rtime = etime(clock,starttime)/i*(nfils-i);
  fprintf(1,'    (noch %4.1f Min) \r', rtime/60);
end;

save results.mat result;                     % Resultat speichern

exectime = etime(clock,starttime);           % Rechenzeit anzeigen
disp(' ')
disp(['... fertig                         ' ...
      '(gebraucht:  ' num2str(exectime/60,2) ' Min)'])
disp(['Erkennungsrate:  ' num2str(ncorr/nfils*100,3) '%'])
