% Uebung 22
%
% Erkennen von einzeln gesprochenen Woertern mit CDHMM
% ----------------------------------------------------
%
% Uebung 22.1
%
% Viterbi-Algorithmus fuer CDHMM
% ------------------------------

% Dieses Matlab-Skript dient zum Testen des Viterbi-Algorithmus fuer CDHMM, 
% der als Matlab-Funktion  [logP,optQ] = log_cont_viterbi_alg(a,mu,sigma,c,X)
% vorliegen muss, mit
%
%   a     Zustandsuebergangswahrscheinlichkeiten eines CDHMM mit
%           N Zustaenden (NxN-Matrix)
%   mu    Mittelwerte der Beobachtungs-Wahrscheinlichkeitsdichten pro 
%         Zustand, Mischkomponente und Element des Beobachtungs- oder 
%         Merkmalsvektors (NxMxD-Matrix) 
%   sigma Varianzen der Beobachtungs-Wahrscheinlichkeitsdichten pro 
%         Zustand, Mischkomponente und Element des Beobachtungs- oder 
%         Merkmalsvektors (NxMxD-Matrix) 
%         (die Kovarianzmatrix wird als diagonal angenommen und in einem
%         Vektor gespeichert -> sigma(i,m,1:D))
%   c     Gewichte der Mischkomponenten (NxM-Matrix) 
%   X     Merkmalssequenz (TxD-Martix, d.h. Sequenz von T Vektoren
%           mit D Elementen) 
%   logP  Logarithmus der Produktionswahrscheinlichkeit (bzw. Likelihood)
%   optQ  optimale Zustandssequenze
%
%         Die Dimensionen der obigen Matrizen sind wie folgt:
%          N = Anzahl Zustaende des CDHMM (inkl. Anfangs- und Endzustand)
%          M = Anzahl Mischkomponenten
%          D = Laenge des Merkmalsvektors
%          T = Laenge der Merkmalssequenz

clear all;

% Testdaten:  CDHMM
                       % Zustaende: N=3, Mischkomponenten: M=1, Dimension: D=1
cdhmm(1).a = [0 1.0 0;  0 0.1 0.9;  0 0 0];
cdhmm(1).mu = zeros(3,1,1);
cdhmm(1).mu(2,1,1) = 1.0; 
cdhmm(1).sigma = zeros(3,1,1);
cdhmm(1).sigma(2,1,1) = 0.1;
cdhmm(1).c = zeros(3,1);
cdhmm(1).c(2,1) = 1.0;

                       % Zustaende: N=3, Mischkomponenten: M=1, Dimension: D=2
cdhmm(2).a = [0 1.0 0;  0 0.5 0.5;  0 0 0];
cdhmm(2).mu = zeros(3,1,2);
cdhmm(2).mu(2,1,:) = [1.0 2.0]; 
cdhmm(2).sigma = zeros(3,1,2);
cdhmm(2).sigma(2,1,:) = [2.0 3.0];
cdhmm(2).c = zeros(3,1);
cdhmm(2).c(2,1) = 1.0;

                       % Zustaende: N=4, Mischkomponenten: M=1, Dimension: D=1
cdhmm(3).a = [0 1.0 0 0;  0 0.3 0.7 0;  0 0 0.2 0.8;  0 0 0 0];
cdhmm(3).mu = zeros(4,1,1);
cdhmm(3).mu(2:3,1,1) = [1.0 2.0]; 
cdhmm(3).sigma = zeros(4,1,1);
cdhmm(3).sigma(2:3,1,1) = [0.05 0.1];
cdhmm(3).c = zeros(4,1);
cdhmm(3).c(2:3,1) = [1.0 1.0];

                       % Zustaende: N=3, Mischkomponenten: M=2, Dimension: D=1
cdhmm(4).a = [0 1.0 0;  0 0.7 0.3;  0 0 0];
cdhmm(4).mu = zeros(3,2,1);
cdhmm(4).mu(2,:,1) = [-0.1 1.0]; 
cdhmm(4).sigma = zeros(3,2,1);
cdhmm(4).sigma(2,:,1) = [2.0 1.0];
cdhmm(4).c = zeros(3,2);
cdhmm(4).c(2,:) = [0.4 0.6];

                       % Zustaende: N=5, Mischkomponenten: M=2, Dimension: D=3
cdhmm(5).a = [0 0.7 0 0.3 0;  0 0.2 0.4 0.3 0.1;  0 0.3 0.2 0 0.5; ...
              0 0.2 0.3 0.4 0.1;  0 0 0 0 0];
cdhmm(5).mu = zeros(5,2,3);
cdhmm(5).mu(2,:,:) = [-0.1 1.0; -0.2 1.5; -0.3 1.8;]'; 
cdhmm(5).mu(3,:,:) = [-0.4 1.0; -0.5 0.5; -0.6 0.8;]'; 
cdhmm(5).mu(4,:,:) = [-0.1 0.0; -0.2 1.0; -0.9 1.8;]'; 
cdhmm(5).sigma = zeros(5,2,3);
cdhmm(5).sigma(2,:,:) = [2.0 1.0; 1.0 1.0; 2.0 2.0]';
cdhmm(5).sigma(3,:,:) = [1.0 1.0; 3.0 1.0; 1.0 2.0]';
cdhmm(5).sigma(4,:,:) = [2.0 1.0; 1.0 1.0; 2.0 1.0]';
cdhmm(5).c = zeros(5,2);
cdhmm(5).c(2,:) = [0.4 0.6];
cdhmm(5).c(3,:) = [0.3 0.7];
cdhmm(5).c(4,:) = [0.5 0.5];


% Testdaten:  Beobachtungssequenzen

X{1}{1} = [1.0];         % Beobachtungssequenz 1 fuer Modell 1
X{1}{2} = [0.0];         % Beobachtungssequenz 2 fuer Modell 1
X{1}{3} = [1.0; 1.0];

X{2}{1} = [0.0 1.0];
X{2}{2} = [0.0 1.0; 0.0 1.0];
X{2}{3} = [0.3 -1.0; 0.5 0.7];

X{3}{1} = [1.0];
X{3}{2} = [1.0; 2.0];
X{3}{3} = [0.9; 1.0; 2.2];

X{4}{1} = [1.0];
X{4}{2} = [1.0; 1.0];
X{4}{3} = [1.0; -1.0; 1.0; 1.1];

X{5}{1} = [0.0 1.0 0.0; 0.0 1.0 0.0];
X{5}{2} = [0.0 1.0 0.0; 0.0 1.0 0.0; 0.0 1.0 0.0];
X{5}{3} = [0.3 -1.0 0; -0.5 0.7 1.0; 0.3 -1.0 0; 0.2 -1.0 -0.5];


% Validierungsdaten

p(1,1) = 1.135409635;        % Modell 1, Beobachtungssequenz 1
p(1,2) = 0.007650329942;     %           Beobachtungssequenz 2
p(1,3) = 0.1432394488;       %           Beobachtungssequenz 3

optQ{1}{1} = [1 2 3];
optQ{1}{2} = [1 2 3];
optQ{1}{3} = [1 2 2 3];

p(2,1) = 0.02141699211;
p(2,2) = 0.000458687551;
p(2,3) = 0.0001476782511;

optQ{2}{1} = [1 2 3];
optQ{2}{2} = [1 2 2 3];
optQ{2}{3} = [1 2 2 3];

p(3,1) = 0.0;
p(3,2) = 1.260442843;
p(3,3) = 0.499782598;

optQ{3}{1} = [];
optQ{3}{2} = [1 2 3 4];
optQ{3}{3} = [1 2 2 3 4];

p(4,1) = 0.09682470989;
p(4,2) = 0.02187505704;
p(4,3) = 0.0004230559921;

optQ{4}{1} = [1 2 3];
optQ{4}{2} = [1 2 2 3];
optQ{4}{3} = [1 2 2 2 2 3];

p(5,1) = 3.512357972e-05;
p(5,2) = 1.416375044e-07;
p(5,3) = 3.619600577e-10;

optQ{5}{1} = [1 2 3 5];
optQ{5}{2} = [1 2 3 3 5];
optQ{5}{3} = [1 2 4 4 3 5];


% Testlauf

if exist('log_cont_viterbi_alg.m') ~= 2,
  disp(['Datei ''log_cont_viterbi_alg.m'' nicht gefunden.']);
  return;
end;

tolerance = 1e-6;

for k = 1:length(cdhmm),
  disp(['Test mit CDHMM' num2str(k) ...
             ':   N=' num2str(size(cdhmm(k).a,1)) ...
               '; M=' num2str(size(cdhmm(k).mu,2)) ... 
               '; D=' num2str(size(cdhmm(k).mu,3)) ])
  for i = 1:length(p(k,:)),
    p_ref = p(k,i);
    optQ_ref = optQ{k}{i};
    [log_p_tst,optQ_tst] = log_cont_viterbi_alg(cdhmm(k).a,cdhmm(k).mu, ... 
                            cdhmm(k).sigma,cdhmm(k).c,X{k}{i});
    p_tst = exp(log_p_tst);
    if abs(p_tst-p_ref)/(p_ref+(p_ref==0)) > tolerance | ...
           length(optQ_tst) ~= length(optQ_ref) | ...
           sum(abs(optQ_tst-optQ_ref)) > 0
      disp(['Fehler mit HMM ' num2str(k) ' und Beobachtungssequenz ' ...
            num2str(i) ':']);
      disp(['Wahrscheinlichkeit']);
      disp(['  vorgegeben: ' num2str(p_ref,10)]);
      disp(['  berechnet:  ' num2str(p_tst,10)]);
      disp(['Optimale Zustandssequenz']);
      disp(['  vorgegeben: [' num2str(optQ_ref) ']']);
      disp(['  berechnet:  [' num2str(optQ_tst) ']']);
      return;
    end;
  end;
end;
disp(['Test ok']);

