function  [c,mfbk] = mfcc(x,fs,nfilt,nceps,mfbk,fLower,fUpper,nfft)
%
%  function  [c,mfbk] = mfcc(x,fs,nfilt,nceps,mfbk,fLower,fUpper,nfft)
%
%  Evaluate the MFCCs (mel frequency cepstral coefficients) according 
%  to the ETSI standard
%
%  Inputs:    x      signal segment
%             fs     sampling frequency 
%             nfilt  number of filters in mel filterbank
%             nceps  number of cepstral coefficients (incl zero term)
%             mfbk   mel-spaced filter bank; if empty, the filter bank
%                    is designed and returned (optional argument)
%             fLower Lower end of lowest filter, relative to fs
%                    (optional argument, default: fLower = 0)
%             fUpper Upper end of highest filter, relative to fs  
%                    (optional argument, default: fUpper = 0.5)
%             nfft   FFT resolution (optional argument, 
%                    default: nfft = length(x))
%
%  Outputs:   c      vector of MFCCs 
%             mfbk   mel-spaced filter bank (optinal output)

x = x(:)';          % make row vector

if nargin < 5       % set default values for optional input arguments
  mfbk = [];
end
if nargin < 6 
  fLower = 0;
end 
if nargin < 7 
  fUpper = 0.5;
end 
if nargin < 8,        
  nfft = length(x);
end

                    % if filter bank not given or empty, create it 
if isempty(mfbk)
  mfbk = melbankm(nfilt,nfft,fs,fLower,fUpper)/2; 
end

if nfft < length(x)
  error('*** FFT resolution must not be smaller than frame size')
end
if sum(abs(x)) > 0,
  X = fft(x,nfft);                       % Fourier spectrum
  n2 = 1+floor(nfft/2);
  Xmel = log(mfbk*abs(X(1:n2))');        % log mel spectrum
  cc = dct(Xmel);                        % mel cepstrum
  c = cc(1:nceps);                       % MFCCs 0...nceps-1
else
  c = zeros(1,nceps);
end
