function [a,b] = discr_baum_welch_training(Xset,N,M,K)
%
%  function [a,b] = discr_baum_welch_training(Xset,N,M,K)
%
%  Estimation of the state transition and observation probabilities 
%  of a discrete density hidden Markov model using the Baum-Welch 
%  algorithm
%
%  Inputs:   Xset   set of observed sequences (training sequences)
%            N      number of states (including start and end state)
%            M      number of observations in an emitting state
%            K      number of training iterations
%
%  Output:   a      estimated transition probabilities as (NxN matrix)
%            b      estimated discrete observation probabilities 
%                   (NxM matrix)

if nargin < 4,
  error('*** missing input arguments');
end

nmin = inf;                       % if the training set has at least one 
for i = 1:length(Xset)            % observation sequence that is shorter 
  if length(Xset{i}) < nmin       % than N-2, the generating HMM cannot 
    nmin = length(Xset{i});       % be a left-right model without skips
  end                             % --> fully connected initial HMM is used
end 
if nmin < N-2    
  error(['*** for a linear HMM with ' num2str(N) ' states, all ' ...
         char(10) 'training sequences must be longer than ' num2str(N-3)]) 
end

[a,b] = init_ddhmm(N,M);         % generate the initial DDHMM


% Training of DDHMM
% -----------------

for k = 1:K       % training iteration      
  
  sum_ns = zeros(1,N);    % initialization 
  sum_nt = zeros(N,N);    
  sum_nb = zeros(N,M);
  

  for s = 1:length(Xset)         % loop over all training sequences
    
                 % estimation of HMM parameters for 1 training sequence
    [ns,nt,nb] = discr_baum_welch_alg(a,b,Xset{s});

    sum_ns = sum_ns + ns;
    sum_nt = sum_nt + nt;
    sum_nb = sum_nb + nb;
  end;
                    % transform expected number of times into probabilities
  for i = 1:N       % --> new estimates of HMM parameters
    a(i,:) = sum_nt(i,:)/sum_ns(i);
    b(i,:) = sum_nb(i,:)/sum_ns(i);
  end;
end;

