function  [ns,nt,nb] = discr_baum_welch_alg(a,b,X)
%
%  function  [ns,nt,nb] = discr_baum_welch_alg(a,b,X)
%  
%  Baum-Welch re-estimation algorithm for a given DDHMM and one 
%  single sequence of discrete observations. The function is intended 
%  to be used for the training of multiple observation sequences and 
%  thus returns the expected number for the basic events 
%  (3 output arguments; see below) 
%
%  Inputs:   a   state transition probabilities (NxN matrix),
%                where a(i,j) denotes the probability to move from 
%                state i to state j
%            b   probabilities of the discrete observations 1...M for 
%                each state (NxM matrix)
%            X   discrete observation sequence (vector of integer 
%                numbers in the range of 1..M).
%
%  Outputs:  ns(i)     expected number of times being in state i
%                      (1xN matrix)
%            nt(i,j)   expected number of times going from state i to j
%                      (NxM matrix)
%            nb(i,k)   expected number of times being in state i with
%                      observation k (NxM matrix)


if nargin < 3,
  error('*** missing input arguments');
end

N = size(a,1);       % number of states (incl. start and end state)
M = size(b,2);       % number of discrete observations
T = length(X);       % length of observations sequence


% Vorwaerts-Wahrscheinlichkeit
% ----------------------------
% Verbundwahrscheinlichkeit, dass sich das HMM zum Zeitpunkt t im 
% Zustand j befindet und die partielle Beobachtungssequenz X(t) 
% erzeugt hat

[forward, alpha] = discr_forward_alg(a, b, X);


% Rueckwaerts-Wahrscheinlichkeit
% ------------------------------
% Produktionswahrscheinlichkeit der noch verbleibenden Beobachtungssequenz
% X(t+1:T) zu einem bestimmten Zeitpunkt t

[backward, beta] = discr_backward_alg(a, b, X);


% Baum-Welch Algorithmus
% ----------------------
% A-posteriori-Wahrscheinlichkeit, dass das HMM zum Zeitpunkt t im 
% Zustand i ist, gegeben die Beobachtungssequenz X und das Modell

gamma = zeros(T,N);
for t = 1:T
  for i = 2:(N-1)
    gamma(t,i) = alpha(t,i)*beta(t,i)/forward;
  end
end

% A-posteriori-Wahrscheinlichkeit des Uebergangs vom Zustand i zum 
% Zustand j zum Zeitpunkt t, gegeben das Modell (a,b) und die 
% Beobachtungssequenz X

xi = zeros(T,N,N);
for t = 1:T-1
  for i = 2:N-1
    for j = 2:N-1
      xi(t,i,j) = alpha(t,i)*a(i,j)*b(j,X(t+1))*beta(t+1,j)/forward;
    end
  end
end

% Erwartete Anzahl Male im Zustand i

ns = zeros(1,N);
for i = 2:N-1
  ns(i) = sum(gamma(:,i));
end
ns(1) = 1;
ns(N) = 1;

% Erwartete Anzahl Uebergaenge vom Zustand i zum Zustand j

nt = zeros(N,N);
for i = 2:N-1
  for j = 2:N-1
    for t = 1:T-1
      nt(i,j) = nt(i,j)+xi(t,i,j);
    end
  end
end

% Erwartete Anzahl Uebergaenge vom Zustand 1 zum Zustand j =
%      erwartete Anzahl Male im Zustand j zum Zeitpunkt t=1

for j = 2:N-1
  nt(1,j) = gamma(1,j);
end

% Erwartete Anzahl Uebergaenge vom Zustand i zum Zustand N =
%             erwartete Anzahl Male im Zustand i zum Zeitpunkt t=T 

for i = 2:(N-1)
  nt(i,N) = gamma(T,i);
end

% Erwartete Anzahl Male im Zustand j mit der Beobachtung X(t)

nb = zeros(N,M);
for j = 2:N-1
  for t = 1:T
    nb(j,X(t)) = nb(j,X(t))+gamma(t,j);
  end
end

