% Uebung 21
%
% Trainieren diskreter HMMs mit dem Baum-Welch-Algorithmus
% -----------------------------------------------------------------
%
% Uebung 21.3
%
% Baum-Welch-Training von DDHMM
% -----------------------------
%
% In dieser Matlab-Funktion koennen im File ddhmms.mat vorhandene DDHMM
% ausgewaehlt werden. Mit jedem der gewaehlten DDHMM wird sodann eine 
% Menge von Beobachtungssequenzen erzeugt und mit diesen ein neues 
% DDHMM trainiert (mit Baum-Welch-Training). Die Parameter des gegebenen
% DDHMM und des neu trainierten DDHMM werden zusammen in einem Fenster
% dargestellt.
%
% Damit dem Programm Argumente uebergeben werden koennen, ist es als 
% Funktion realisiert, welche die folgenden (optionalen) Argumente
% aufweist:
%
%   Ind        Indizes der DDHMM im File ddhmms.mat, fuer die ein Training
%              durchgefuehrt werden soll  (default = [1 2 3 4 6])
%   S          Anzahl zu erzeugende Trainingssequenzen pro DDHMM
%              (default = 50)
%   K          Anzahl Trainingsiterationen (default = 5)
%   initRand   Initialisierung des Zufallsgenerators 
%              (default = 0, d.h. keine Initialisierung)

function [] = ueb21_3(Ind,S,K,initRand)


%----- Eingangsargumente testen und ev. Default-Werte setzen -----

nerr = 0;
if nargin < 1 | isempty(Ind),  Ind = [1,2,3,4,6];  end
if nargin < 2,  S = 50;  else  nerr = nerr+checkarg('S',S,[1 10000]);  end
if nargin < 3,  K = 10;  else  nerr = nerr+checkarg('K',K,[1 1000]);  end
if nargin < 4
  initRand = 0;
else
  nerr = nerr+checkarg('initRand',initRand,[0 1]);
end
if nerr > 0,  return;  end


%----- Einstellungen -----

blnPlot = 1; % Graph. Darstellung der trainierten DDHMM
blnInfo = 1; % Konsolen-Information zum Training (enable=1/disable=0)
blnSave = 1; % Speichern der trainierten DDHMM im File ddhmms_trained.mat


%----- DDHMM laden und Auswahl ueberpruefen -----

load ddhmms.mat;         % DDHMM in Variable `ddhmm`

if ~isempty(Ind(Ind > length(ddhmm))) | ~isempty(Ind(Ind < 1)) 
  disp(['Indizes der gewaehlten DDHMM sind nicht zwischen 1 und ' ...
        num2str(length(ddhmm))]);
  return;
end;



% Erzeugen der Trainingssequenzen
% --------------------------------
%
% Die Trainingssequenzen werden in der Variable 'trainSeq' gesammelt,
% welche die folgende Struktur hat:
%
% trainSet{m}.X{s}   Trainingssequenzen s = 1..S fuer die 
%                    Modelle m = 1..length(Ind)

if initRand,
  rand('state',0);
end;
NumOfModels = length(Ind);
trainSeq = {};
for m = 1:NumOfModels,
  for s = 1:S,
    % Trainingssequenz generieren
    [X,Q] = gen_discr_obs_seq(ddhmm(Ind(m)));
    trainSet{m}.X{s} = X;
  end
end

% Training der DDHMM
% ------------------
%
% Die einzelnen trainierten Modelle werden der Variable 'ddhmm2'
% zugewiesen, welche die folgende Struktur aufweist:
%
% ddhmm.name    Name des Modells; Lautfolge (character array) 
% ddhmm.a       Zustandsuebergangs-Wahrscheinlichkeiten  (NxN array)
% ddhmm.b       Beobachtungswahrscheinlichkeiten  (NxM array)

elapsedTime = [];   % Benoetigte Rechenzeit fuer das Training
for m = 1:NumOfModels,
  if blnInfo,
    tic;    % Start des Timers
    disp(['Training Modell ',num2str(Ind(m)),' ( ', ...
          ddhmm(Ind(m)).name,' ) ...']);
  end;
                                                  % Initialisierung
  N = size(ddhmm(Ind(m)).a,1);     % Anzahl Zustaende
  M = size(ddhmm(Ind(m)).b,2);     % Anzahl Beobachtungen
  
                                                    % Baum-Welch-Training
  [a,b] = discr_baum_welch_training(trainSet{m}.X,N,M,K);
  ddhmm2(m).name = ddhmm(Ind(m)).name;
  ddhmm2(m).a = a;
  ddhmm2(m).b = b;
  
  if blnInfo,
    elapsedTime(m) = toc;      % Stoppen des Timers
    disp(['Rechenzeit: ',num2str(elapsedTime(m)),' Sek.']);
  end;
  
  
%----- Graphische Anzeige des Trainingsresultates -----

  if blnPlot,
    hnd = [];
    hFig = figure;
    set(hFig,'position',[60 720 1120 420]);   % [left,bottom,width,height]
    set(hFig,'menubar','none');
    clf;
    colorOrig = 'b'; colorTrain = 'r';
    
    for i = 2:N-1,         % Anzeige der Beobachtungswahrscheinlichkeiten
      subplot(max(3,N-2),2,2*(i-1)-1);
      hold on
      dpdf1 = [ddhmm(Ind(m)).b(i,:)',zeros(M,1)];
      hB = bar(dpdf1,colorOrig); set(hB,'edgecolor',colorOrig);
      dpdf2 = [zeros(M,1),ddhmm2(m).b(i,:)'];
      hB = bar(dpdf2,colorTrain); set(hB,'edgecolor',colorTrain);
      dpdf3 = [zeros(M,2)]; hB=bar(dpdf3,'g'); set(hB,'edgecolor','g');
      line([0,M+1],[0,0],'color','k');
      hold off
      grid;
      axis([0.1,M+0.9,0,1]);
      set(gca,'xtick',1:M)
      delta_dpdf = sum(abs(ddhmm(Ind(m)).b(i,:)-ddhmm2(m).b(i,:))/M);
      hnd = [hnd,text(0.3,0.8,['Mittlere Abweichung: ',num2str(delta_dpdf)])];
      hnd = [hnd,ylabel(ddhmm(Ind(m)).name(i-1))];
    end;
    subplot(max(3,N-2),2,1);
    hnd = [hnd,title(['Modell ',num2str(Ind(m)), ...
                    ' ( ',ddhmm(Ind(m)).name, ...
                    ' ) - Beobachtungswahrscheinlichkeiten'])];
                             
    subplot(3,2,2); axis off;     % Anzeige der uebergangswahrscheinlichkeiten
    hnd = [hnd,title(['Modell ',num2str(Ind(m)), ...
                    ' ( ',ddhmm(Ind(m)).name, ...
                    ' ) - Uebergangswahrscheinlichkeiten'])];
    hnd = [hnd,text(0,0.85,'Originales Modell')];
    set(hnd(end),'color',colorOrig);
    for i = 1:N,
      hnd = [hnd,text((i-1)*0.2,0.7, ...
            strjust(num2str(ddhmm(Ind(m)).a(:,i)),'left'),'vertical','top')];
      set(hnd(end),'color',colorOrig);
    end;
    subplot(3,2,4); axis off;
    hnd = [hnd,text(0,0.85,'Trainiertes Modell')];
    set(hnd(end),'color',colorTrain);
    for i = 1:N,
      hnd = [hnd,text((i-1)*0.2,0.7, ...
            strjust(num2str(ddhmm2(m).a(:,i)),'left'),'vertical','top')];
      set(hnd(end),'color',colorTrain);
      delta_trans(i) = sum(abs(ddhmm(Ind(m)).a(i,:)-ddhmm2(m).a(i,:))/N);
    end;
                  % Anzeige der Anzahl Trainingssequenzen und -iterationen
    subplot(3,2,6); axis off;
    hnd = [hnd,text(0,0.6,['Anzahl Trainingssequenzen: ',num2str(S)])];
    hnd = [hnd,text(0,0.4,['Anzahl Trainingsiterationen: ',num2str(K)])];
   
    set(hnd,'rotation',0,'fontweight','bold','fontsize',10);
    drawnow
  end
end


%----- Speichern der trainierten Modelle -----

if blnSave,
  save ddhmms_trained.mat ddhmm2 Ind;
end;



%----- Lokale Funktion -------------------------------------------------

function  nerr = checkarg(xstr,x,r);

%  Test ob das Argument `x` numerisch und im Bereich r(1) ... r(2) ist

if isempty(x) | ~isnumeric(x) | x ~= round(x) | x < r(1) | x > r(2)
  disp(['*** Argument `' xstr '` der Funktion  ' ...
         'ueb21_3(Ind,S,K,initRand)  ist ungueltig'])
  nerr = 1;
else
  nerr = 0;
end
