function  [a,b] = init_ddhmm(N,M,mtype)
%
%  function  [a,b] = init_ddhmm(N,M,mtype)
%
%  Initialize a DDHMM with N states and M discrete observations
%
%  Inputs:   N   number of states (incl. non-emitting start and end states)
%            M   number of discrete observations
%        mtype   type of model:  'standard' (fully connected) 
%                                'left-right'  
%                                'linear' (left-right without skips)
%                default:  'linear'
%
%  Outputs:  a   transition probabilities (NxN matrix)
%            b   observation probabilities (NxM matrix) 

if nargin < 3,
  mtype = 'linear';
end  
if nargin < 2,
  error('*** missing input arguments');
end;


%----- Initialization of transition matrix -----

a = zeros(N,N);
switch mtype       
  case 'standard'                % fully connected model
    for i = 1:N-1        
      a(i,2:N) = 1;
      if i == 1
        a(1,N) = 0;       % exclude zero length observation sequences   
      end
      a(i,:) = a(i,:)/sum(a(i,:));
    end
  case 'left-right'
    for i = 1:N-1        
      a(i,i:N) = 1;
      if i == 1
        a(1,1) = 0;
        a(1,N) = 0;       % exclude zero length observation sequences   
      end
      a(i,:) = a(i,:)/sum(a(i,:));
    end
  case 'linear'                  % left-right without skips
    a(1,2) = 1;      % transition prob from start state to state 2
    for i = 2:N-1    % transition probs from emitting states 2 to N-1
      a(i,i:i+1) = [0.75 0.25];
    end
  otherwise
    error(['*** invalid model type:  ' mtype])
end  


%----- Initialization of discrete probability density matrix -----

b = zeros(N,M);
c = 0;            % c = 0:  uniform distribution
                  % c = 1:  biased random distribution
for i = 2:N-1
  b(i,:) = 1+c*rand(1,M);
  b(i,:) = b(i,:)/sum(b(i,:)); 
end

