function  [p,optQ] = discr_viterbi_alg(a,b,X)
%
%  function  [p,optQ] = discr_viterbi_alg(a,b,X)
% 
%  This implementation of the Viterbi algorithm evaluates for the 
%  given observation sequence `X` and the discrete density HMM (given 
%  by `a` and `b`) the optimal state sequence `optQ`, i.e. the state 
%  sequence with the highest joint probability P(Q,X|HMM). If the 
%  HMM is not able to generate the given observation sequence, the 
%  probability is zero and the optimal state sequence is empty.
%
%  Inputs:   a   state transition probabilities (NxN matrix),
%                where a(i,j) denotes the probability to move from 
%                state i to state j
%            b   probabilities of the discrete observations 1...M for 
%                each state (NxM array)
%            X   discrete observation sequence (vector of integer 
%                numbers in the range of 1..M).
%
%  Outputs:  p     joint probability of the optimal state sequence 
%                  and the given observation sequence `X`
%            optQ  optimal state sequence (vector with 2 elements 
%                  more than `X`)

if nargin ~= 3,
  error('*** incorrect number of input arguments');
end;
if size(a,1) ~= size(a,2)
  error(['*** input argument a must be a quadratic matrix']);
end;
if size(a,1) ~= size(b,1)
  error(['*** input arguments a and b must have the same number of rows']);
end;

N = size(a,1);
T = length(X);

delta = zeros(T,N);
psi = zeros(T,N);

for j = 2:(N-1)                                 % initialization
  delta(1,j) = a(1,j) * b(j, X(1));
end
psi(1,:) = ones(1,N);

for t = 2:T                                     % recursion 
  for j = 2:(N-1)
    [delta(t,j) psi(t,j)] = max(delta(t-1,:).*a(:,j)' * b(j,X(t)));
  end
end

[p psi(T,N)] = max(delta(T,:).*a(:,N)');         % termination

if p == 0
  optQ = [];                        % return empty state sequence 
else
  optQ = zeros(1,T);                % backward search of optimal path
  optQ(T) = psi(T,N);
  for t = T-1:-1:1
    optQ(t) = psi(t+1, optQ(t+1));
  end
  optQ = [1 optQ N];
end

