function [ p, beta ] = discr_backward_alg( a, b, X )
%
%  function  [ p, beta ] = discr_backward_alg(a,b,X)
% 
%  The backward algorithm evaluates the probability `p` that the 
%  discrete density HMM given by `a` and `b` produces the 
%  observation sequence `X`. It also returns the intermediate
%  probabilities `beta` for the HMM to be at time `t` in a given 
%  state and producing the partial observation sequence 
%  X(t+1), X(t+2),...,X(T).
%
%  Inputs:   a   state transition probabilities (NxN matrix),
%                where a(i,j) denotes the probability to move from 
%                state i to state j
%            b   probabilities of the discrete observations 1...M for 
%                each state (NxM array)
%            X   discrete observation sequence (vector of integer 
%                numbers in the range of 1..M).
%
%  Output:   p      production probability
%            beta   joint backward probability(TxN matrix),
%                   where beta(t,i) denotes the joint probability that the 
%                   HMM is in state Si at time t and will produced the 
%                   partial observation sequence X(t+1), X(t+2),...,X(T).

if nargin ~= 3,
  error('*** incorrect number of input arguments');
end;
if size(a,1) ~= size(a,2)
  error(['*** input argument a must be a quadratic matrix']);
end;
if size(a,1) ~= size(b,1)
  error(['*** input arguments a and b must have the same number of rows']);
end;

N = size(a,1);       % number of states (incl. start and end state)
T = length(X);       % length of observations sequence


beta = zeros(T,N);
for i = 2:N-1                                   % initialization
    beta(T,i) = a(i,N);
end
for t = T-1:-1:1                                % recursion
    for i = 2:N-1
        for j = 2:N-1
            beta(t,i) = beta(t,i)+a(i,j)*b(j,X(t+1))*beta(t+1,j);
        end
    end
end
p = 0;                                   % termination
for j = 2:N-1
    p = p+a(1,j)*b(j,X(1))*beta(1,j);
end

end

