function [hFig,ButtonPressed] = display_gui(hFig,DDHMMs,selModels, ...
                                obsSeq,stateSeq,prFwd,optStateSeq,prVit,cfg)
%
%  function:  [hFig,ButtonPressed] = display_gui(hFig,DDHMMs,SelModels, ...
%                   obsSeq,stateSeq,prFwd,optStateSeq,prVit,cfg)
%
%  Create (or update if already existent) a graphical user interface (GUI) 
%  that enables to display HMM observation and state sequences. 
%  Optionally the recognizing HMM will be marked.
%
%
%  Mandatory input arguments:
%
%  hFig         Figure handle; if handle is empty or figure not existent, 
%               a new figure is created; otherwise the existing figure
%               is updated. 
%
%  DDHMMs       Vector of discrete density HMMs. Each element representing 
%               an N state HMM with M discrete observations has to agree 
%               with the following structure definition:
%               DDHMM.name   name of HMM (or phone string)
%               DDHMM.a      transition probabilities (NxN matrix)
%               DDHMM.b      observation probabilities (NxM matrix)
%
%  selModels    Vector containing the indices of the HMMs in DDHMMs 
%               that are displayed in the GUI. 
%
%
%  Optional input arguments:
%
%  obsSeq       Observation sequence (vector)
%
%  stateSeq     State sequence (vector that is two elements longer than 
%               `obsSeq`)
%
%  prFwd        Forward probabilities (vector) of the selected models 
%               in `selModels`.
%
%  optStateSeq  Optimal state sequence
%
%  prVit        Probability along the optimal path from Viterbi algorithm
%
%  cfg          Optional configuration structure with the following 
%               fields (other fields will not be evaluated):
%
%               cfg.showRecModels:  Show/Hide the recognizing models
%                     1 = show (default), 0 = hide
%
%               cfg.showStartStop:  Show/Hide start/stop button 
%                     1 = show, 0 = hide (default)
%
%
%  Output arguments:
%
%  hFig         Handle to current figure (for subsequent calls)
%
%  ButtonPressed  Number of the currently activated button according
%                 to the array of 'selModels' (a positive value)
%                 or a number of one of the functional buttons.
%                  -1: 'Exit' the application
%                  -2: 'Start' run mode
%                  -3: 'Stop' run mode


%----- check input arguments -----

if nargin < 9,
  cfg.showRecModels = 1;
  cfg.showStartStop = 0;
  if nargin < 8,
    prVit = [];
    if nargin < 7,
      optStateSeq = [];
      if nargin < 6,
        prFwd = [];
        if nargin < 5,
          stateSeq = [];
          if nargin < 4,
            obsSeq = [];
            if nargin < 3,
              error('*** function needs minimum 3 input arguments');
            end;
          end;
        end;
      end;
    end;
  end;
end;


%----- figure settings (create new or used existing figure) -----

if isempty(hFig) | ~ishandle(hFig)       % test if GUI figure exists

  if isempty(DDHMMs)                  % test input arguments
    error('*** no DDHMMs selected');
  end
  for i = 1:length(selModels)
    if selModels(i) < 1 | selModels(i) > length(DDHMMs) 
      error(['*** selected model ' num2str(selModels(i)) ' not in DDHMMs']);
    end
  end
  
  hFig = figure;                         %----- create a new GUI figure -----
  set(hFig,'position',[60 120 1120 320]);   % [left,bottom,width,height]
  set(hFig,'menubar','none');
  clf;
                                    % init all figure settings and user data
  fdata = init_settings(selModels,hFig,cfg);
  fdata.AxesHandle = axes;
  axis off;
                                     
  fdata = add_elements(fdata,DDHMMs);    % add buttons to figure

else                                    %----- use existing figure -----
  if ~ishandle(hFig)
    error(['*** non existent figure handle:  ' num2str(hFig)]);
  end;
  fdata = get(hFig,'userdata');         % get settings and fdata
  display_sequences(fdata,0);           % remove displayed sequences

  if sum(prFwd) > 0,                    % set the recognizing models
    recModFwd = find(prFwd==max(prFwd));
  else
    recModFwd = [];
  end;
  if sum(prVit) > 0,
    recModVit = find(prVit==max(prVit));
  else
    recModVit = [];
  end;

  for i = 1:length(recModFwd)           % update counters of rec models
    fdata.countRecFwd(recModFwd(i)) = fdata.countRecFwd(recModFwd(i))+1;
                                             % mark recognizing model
    set(fdata.hRecButtons(recModFwd(i)),'background',fdata.ColorBtnMarked);
  end
  for i = 1:length(recModVit)
    fdata.countRecVit(recModVit(i)) = fdata.countRecVit(recModVit(i))+1;
                                             % mark recognizing model
    set(fdata.hRecButtons(recModVit(i)),'background',fdata.ColorBtnMarked)
  end
  display_sequences(fdata,1,obsSeq,stateSeq,prFwd,optStateSeq,prVit);
                               % enable only the recog model that 
  if ~isempty(optStateSeq)    % has generated the optimal state seq 
    for i = 1:length(fdata.hRecButtons)  
      if i == str2num(get(fdata.hCurrentButton,'tag'))
        set(fdata.hRecButtons(i),'enable','on')
      else
        set(fdata.hRecButtons(i),'enable','off')
      end
    end
  end
end


%----- wait for button being pressed (if not in run mode) -----

if strcmp(get(fdata.hStartStopBtn,'tag'),'-3')   % if in run mode
  pause(0.3)
  fdata.hPrevButton = fdata.hCurrentButton;
  ButtonPressed = str2num(get(fdata.hCurrentButton,'tag'));
else
  if get(fdata.hStartStopBtn,'userdata') == 1    % if stop has been pressed
    while 1                                      % clear all pressed buttons
      h = findobj(get(hFig,'children'),'type','uicontrol','userdata',1);
      if isempty(h)
        break
      end
      set(h(1),'userdata',0);
    end 
  end
  ButtonPressed = 0;   
end

while ButtonPressed == 0                         % wait for button pressed
  pause(0.1)
                                          % if figure window has been closed 
  if sum(findobj(0,'type','figure') == hFig) == 0
    ButtonPressed = -1;                      
    return
  end

  h = findobj(get(hFig,'children'),'type','uicontrol','userdata',1);
  if ~isempty(h)
    set(h(1),'userdata',0);
    if str2num(get(h(1),'tag')) == -1            % exit button pressed
      ButtonPressed = -1;                      
      close(hFig);
      return
    elseif str2num(get(h(1),'tag')) == -3        % start has been pressed
      ButtonPressed = str2num(get(fdata.hCurrentButton,'tag'));
    else                                         % gen button has been pressed 
      fdata.hPrevButton = fdata.hCurrentButton;
      fdata.hCurrentButton = h;
      ButtonPressed = str2num(get(fdata.hCurrentButton,'tag'));
    end 

    if cfg.showStartStop == 1                    % enable start/stop button
      if ButtonPressed > 0
        set(fdata.hStartStopBtn,'enable','on');
      end
    end
  end
end;


%----- mark generating button and adjust counters -----
                       
for i = 1:fdata.Number                            % ummark all buttons
  set(fdata.hGenButtons,'background',fdata.ColorBtnUnmarked);
  set(fdata.hRecButtons,'background',fdata.ColorBtnUnmarked);
end
                                          % mark button of generating model
set(fdata.hCurrentButton,'background',fdata.ColorBtnMarked);

if isempty(fdata.hPrevButton) | fdata.hPrevButton ~= fdata.hCurrentButton   
  fdata.countGen = 0;                     % reset counter of gen model
  for i = 1:fdata.Number                  % reset counters of rec models
    fdata.countRecFwd(i) = 0;
    fdata.countRecVit(i) = 0;
  end
end
fdata.countGen = fdata.countGen+1;
set(hFig,'userdata',fdata)         % store as property 'userdata' in figure




%=========================================================================
%  local functions
%=========================================================================


function fdata = init_settings(sel_models,hfig,cfg);
%
%  Inititialze all parameters used for graphics and the 
%  data (e.g. state seq., observation seq. etc)

fdata.FigureHandle = hfig;
fdata.AxesHandle = [];
fdata.MarginX = [0.0,0.9];
fdata.MarginY = [0.0,1.0];
fdata.Number = length(sel_models);
fdata.Indizes = sel_models;
fdata.ColorBtnMarked = [0.7 0.0 0.4];
fdata.ColorBtnUnmarked = [0.8 0.85 0.8];
fdata.ColorLetters = 'k';
fdata.ColorTrue = [0.0 0.6 0.0];
fdata.ColorFalse = [0.7 0.0 0.0];
fdata.FontWeight = 'bold';
fdata.FontSizeLetters = 11;
fdata.FontSizeTitle = 14;

fdata.GenTitleStr = 'Generierende HMM';
fdata.GenTitleCoords = [0.55,1.0];
fdata.GenStartCoords = [0.15,0.79];
fdata.hGenTitle = [];
fdata.hGenButtons = [];

fdata.RecTitleStr = 'Erkennende HMM';
fdata.RecTitleCoords = [0.55,0.45];
fdata.RecStartCoords = [0.15,0.35];
fdata.hRecTitle = [];
fdata.hRecButtons = [];
fdata.showRecModels = cfg.showRecModels;

fdata.hStartStopBtn = [];
fdata.showStartStop = cfg.showStartStop;
fdata.hExitBtn = [];

fdata.hCurrentButton = [];
fdata.hPrevButton = [];




function fdata = add_elements(fdata,Models)
%
%  adds all necessary graphical elements (e.g. buttons) to figure
%
%
%  Models          Array of HMMs (see above)
%
%  selModels       vector holding the indices of models to be displayed
%
%  FigHnd          According figure handle.
%
%  showRecModels   = 0   display buttons of generating models only 
%                  = 1   display buttons of generating and recognizing models
%
%  showStartStop   = 0   do not display start/stop button 
%                  = 1   display start/stop button 
%                            
%  hBtn            Button handles:
%                    hBtn.RecBtns    handles to rec model buttons
%                    hBtn.GenBtns    handles to gen model buttons
%                    hBtn.ExitBtn    handle to exit button
%                    hBtn.StartBtn   handle to start/stop button
%

fdata = add_buttons(fdata,Models,'gen');    % buttons for generating models
fdata = add_buttons(fdata,Models,'rec');    % buttons for recognizing models

btnLength = 0.07; % in percent of figure size      % control buttons
btnHeight = 0.1;  % in percent of figure size

                                                   % exit button
fdata.hExitBtn = uicontrol( ...
    'style','push', ...
    'units','norm', ...
    'pos',[1-btnLength,0,btnLength,btnHeight], ...
    'background',fdata.ColorBtnUnmarked, ...
    'foreground',fdata.ColorLetters, ...
    'fontweight',fdata.FontWeight, ...
    'fontsize',fdata.FontSizeLetters, ...
    'string','Exit', ...
    'tag','-1', ...
    'userdata',0, ...
    'callback','set(gcbo,''userdata'',1);');

                                                % start/stop button
if fdata.showStartStop,
  callback_def = ['if strcmp(get(gcbo,''string''),''Start >''),' ...
                       'strBtn=''Stop ||'';' ...
                       'strTag=''-3'';' ...
                       'else,' ...
                       'strBtn=''Start >'';' ...
                       'strTag=''-2'';' ...
                       'end;' ...
                       'set(gcbo,''userdata'',1);' ...
                       'set(gcbo,''string'',strBtn,''tag'',strTag);'];
  fdata.hStartStopBtn = uicontrol( ...
      'style','push', ...
      'units','norm', ...
      'pos',[1-btnLength,0.1,btnLength,btnHeight], ...
      'background',fdata.ColorBtnUnmarked, ...
      'foreground',fdata.ColorLetters, ...
      'fontweight',fdata.FontWeight, ...
      'fontsize',fdata.FontSizeLetters, ...
      'string','Start >', ...
      'tag','-2', ...
      'enable','off', ...
      'userdata',0, ...
      'callback',callback_def);
end;



function fdata = add_buttons(fdata,models,btyp);
%
%  put buttons into figure:   btyp = 'gen'  -->  generating models 
%                                  = 'rec'  -->  recognizing models

if strcmp(btyp,'gen')                                           % title
  h = text(fdata.GenTitleCoords(1),fdata.GenTitleCoords(2),fdata.GenTitleStr);
  fdata.hGenTitle = h;
  vis_ctrl = 'on';
  btnX = fdata.GenStartCoords(1);   % lower left corner of the first button
  btnY = fdata.GenStartCoords(2);
else
  h = text(fdata.RecTitleCoords(1),fdata.RecTitleCoords(2),fdata.RecTitleStr);
  fdata.hRecTitle = h;
  if fdata.showRecModels 
    vis_ctrl = 'on';
  else
    vis_ctrl = 'off';
  end
  btnX = fdata.RecStartCoords(1);
  btnY = fdata.RecStartCoords(2);
end
set(h,'HorizontalAlignment','center');
set(h,'FontWeight',fdata.FontWeight);
set(h,'FontSize',fdata.FontSizeTitle);
set(h,'Visible',vis_ctrl);

                % button dimensions (depending on the number of buttons)
btnHt = 0.1;
btnWdth = (fdata.MarginX(2)-btnX)/fdata.Number;
btnDist = 1.1*btnWdth;  % distance from one lower left corner to the next one
if strcmp(btyp,'gen')                                           % title
  enable_ctrl = 'on';
else
  enable_ctrl = 'inactive';
end
for i = 1:fdata.Number,
  hButtons(i) = uicontrol(...
      'style','push', ...
      'units','norm', ...
      'pos',[btnX+(i-1)*btnDist btnY btnWdth btnHt], ...
      'background',fdata.ColorBtnUnmarked, ...
      'foreground',fdata.ColorLetters, ...
      'fontweight',fdata.FontWeight, ...
      'fontsize',fdata.FontSizeLetters, ...
      'string',['DDHMM(',num2str(fdata.Indizes(i)), ...  
                ')  [',models(fdata.Indizes(i)).name,']'], ...
      'tag',num2str(i), ...
      'visible',vis_ctrl, ...
      'enable',enable_ctrl, ...
      'interruptible','off', ...
      'userdata',0, ...
      'callback','set(gcbo,''userdata'',1);');
end;
if strcmp(btyp,'gen') 
  fdata.hGenButtons = hButtons;
else
  fdata.hRecButtons = hButtons;
end 


function  display_sequences(fdata,dispStrings,obsSeq,stateSeq,prFwd, ...
              optStateSeq,prVit);
%
%  displays the provided sequences in several rows of numbers
%

if ~dispStrings
            % clear sequence: clear all text boxes (except the two titles)
  handles = get(fdata.AxesHandle,'children');
  if length(handles) > 2,
    delete(handles(1:length(handles)-2));
  end;
  
else
  txtX = -0.15;                         % local display settings
  txtY = 0.7;
  xoffset = 0.05;
  yoffset = 0.08;
  
  txtPhrase2 = 'Zustandssequenz';       % descriptions / subtitles
  txtPhrase3 = 'Beobachtungssequenz';
  
  txtPhraseFwdProb = 'Forward-Wahrscheinlichkeit';
  txtPhraseVitProb = 'Viterbi-Wahrscheinlichkeit';
  txtPhraseFwdRec = 'Forward-Erkennung [%]';
  txtPhraseVitRec = 'Viterbi-Erkennung [%]';
  txtPhraseVitSeq = 'Opt. Zustandssequenz';
  
  lenStateSeq = length(stateSeq);
  lenObsSeq = length(obsSeq);
  if (lenStateSeq+lenObsSeq > 0) & (lenStateSeq ~= lenObsSeq+2) 
    error(['*** length of state seq (' num2str(lenStateSeq) ...
           ') and observation seq (' num2str(lenObsSeq) ') do not match'])
  end
  
                     % display sequences of states and observations 
  if lenStateSeq > 0
    text(txtX,txtY,txtPhrase2);
  end
  if lenObsSeq > 0 
    text(txtX,txtY-yoffset,txtPhrase3);
  end
  
  t = 1;                                     % non-emitting start state
  xstep = 1/(lenStateSeq+1);
  if lenStateSeq > 0
    text(xoffset+t*xstep,txtY,num2str(stateSeq(t)));
  end
                                                % emitting states
  t = 2:(lenStateSeq-1);
  text(xoffset+t.*xstep,txtY.*ones(size(t)), ...
               strjust(num2str(stateSeq(t)'),'left'));
  text(xoffset+t.*xstep,(txtY-yoffset).*ones(size(t)), ...
               strjust(num2str(obsSeq(t-1)'),'left'));

  t = lenStateSeq;                           % non-emitting end state
  text(xoffset+t*xstep,txtY,num2str(stateSeq(t)));
  
          % display recognition probs and rates if optimal state seq empty
  txtY = txtY-0.2;
  scaling = 100; 
  if isempty(optStateSeq) 
    if fdata.showRecModels & (length(prFwd) > 0)
                                         % display forward probabilities
      text(txtX,txtY-4*yoffset,txtPhraseFwdProb);
      for i = 1:fdata.Number
        text(fdata.RecStartCoords(1)+(i-1.1)/fdata.Number, ...
             txtY-4*yoffset,strjust(num2str(prFwd(i)),'left'));
      end
                          % display recognition rates of forward algorithm
      text(txtX,txtY-6*yoffset,txtPhraseFwdRec);
      for i = 1:fdata.Number
        rt = fdata.countRecFwd(i)/fdata.countGen;
        text(fdata.RecStartCoords(1)+(i-1.1)/fdata.Number,txtY-6*yoffset, ...
               strjust(num2str(rt*scaling,4),'left'));
      end
    end
 
                                         % show Viterbi probabilities
    if fdata.showRecModels & (length(prVit) > 0)
      text(txtX,txtY-5*yoffset,txtPhraseVitProb);
      for i = 1:fdata.Number
        text(fdata.RecStartCoords(1)+(i-1.1)/fdata.Number, ...
             txtY-5*yoffset, strjust(num2str(prVit(i)),'left'));
      end
                          % display recognition rates of Viterbi algorithm
      text(txtX,txtY-7*yoffset,txtPhraseVitRec);
      for i = 1:fdata.Number
        rt = fdata.countRecVit(i)/fdata.countGen;
        text(fdata.RecStartCoords(1)+(i-1.1)/fdata.Number,txtY-7*yoffset, ...
               strjust(num2str(rt*scaling,4),'left'));
      end
    end
  end

  % show optimal state sequence from Viterbi algorithm 
  % (the differences between the state seq. of the generating model and
  % the optimal state seq. from the Viterbi algorithm are high-lighted)
  diffStates = [];
  if length(optStateSeq) > 0,
    text(txtX,txtY-5*yoffset,txtPhraseVitSeq);
    Len = length(optStateSeq);
                      
    xstep = 1/(Len+1);                  % display optimal state sequence
    n = 1:Len;
    resH(n) = text(xoffset+n.*xstep,(txtY-5*yoffset).*ones(size(n)), ... 
                 strjust(num2str(optStateSeq(n)'),'left'));

    for n = 1:Len,       % compare original and optimal state sequences
      if n <= length(stateSeq),
        if optStateSeq(n) == stateSeq(n),
          set(resH(n),'color',fdata.ColorTrue);
        else
          set(resH(n),'color',fdata.ColorFalse);
	  diffStates = [diffStates n];
        end;
      else
        set(resH(n),'color',fdata.ColorLetters);
      end;
    end;
  end;
                                     % set font properties and visibility
  h = get(fdata.AxesHandle,'children');
  hh = h(1:length(h)-2);
  if length(hh) > 0
    set(hh,'fontweight',fdata.FontWeight);
    set(hh,'fontsize',fdata.FontSizeLetters);
  end;

  waittime = 0.;   % if special event, wait to allow clicking of stop button
  if length(prFwd) > 0 
    [x,idxf] = sort(prFwd);
    if str2num(get(fdata.hCurrentButton,'tag')) ~= idxf(end)
      waittime = 1;
    end
  end
  if length(prVit) > 0 
    [x,idxv] = sort(prVit);
    if str2num(get(fdata.hCurrentButton,'tag')) ~= idxv(end)
      waittime = 1;
    end
  end
                        % if recog results from Forward and Viterbi differ 
                        % set stop button to halt the program
  if exist('idxf','var') & exist('idxv','var')
    if idxf(end) ~= idxv(end)
      disp(['Sequences generated:  ' num2str(fdata.countGen)])
      disp(['Forward algorithm:  Model ' num2str(idxf(end))])
      disp(['Viterbi algorithm:  Model ' num2str(idxv(end))])
      set(fdata.hStartStopBtn,'string','Start >','tag','-2','userdata',1)
    end
  end
  
  % highlight differences between optimal and original state sequence
  if length(diffStates) > 0,
    waittime = 1;
    cycle = ['on ';'off';'on ';'off';'on '];
    for k = 1:length(cycle),
      set(resH(diffStates),'visible',deblank(cycle(k,:)));
      pause(0.1);
    end;
  end;
  pause(waittime);
  
end;




