function  [p, alpha] = discr_forward_alg(a,b,X)
%
%  function  [p, alpha] = discr_forward_alg(a,b,X)
% 
%  The forward algorithm evaluates the probability `p` that the 
%  discrete density HMM given by `a` and `b` produces the 
%  observation sequence `X`. It also returns the intermediate
%  probabilities `alpha` for the HMM to be at time `t` in a given 
%  state and having produced the partial observation sequence 
%  X(1), X(2),...,X(t).
%
%  Inputs:   a   state transition probabilities (NxN matrix),
%                where a(i,j) denotes the probability to move from 
%                state i to state j
%            b   probabilities of the discrete observations 1...M for 
%                each state (NxM array)
%            X   discrete observation sequence (vector of integer 
%                numbers in the range of 1..M).
%
%  Output:   p       production probability
%            alpha   joint forward probability(TxN matrix),
%                    where alpha(t,i) denotes the probability that the HMM
%                    is in state Si at time t and has produced the 
%                    partial observation sequence X(1), X(2),...,X(t)

if nargin ~= 3,
  error('*** incorrect number of input arguments');
end;
if size(a,1) ~= size(a,2)
  error(['*** input argument a must be a quadratic matrix']);
end;
if size(a,1) ~= size(b,1)
  error(['*** input arguments a and b must have the same number of rows']);
end;

N = size(a,1);
T = length(X);

p = 0;
alpha = zeros(T,N);

for j = 2:(N-1)                           % initialization
  alpha(1,j) = a(1,j) * b(j,X(1));
end

for t = 2:T                               % recursion
  for j = 2:(N-1)
    alpha(t,j) = sum(alpha(t-1,:).*a(:,j)') * b(j,X(t));   % (sum over i)
  end
end

for i = 2:(N-1)                           % termination
  p = p + alpha(T,i) * a(i,N);
end


