function [phrbnd,phrtypes] = phrasing_sol(tree);
%  
%  [phrbnd,phrtypes] = phrasing(tree);
% 
%  Assignment of phrase boundary positions and strengths. 
%
%  Input:    tree      contains an annotated syntax tree with words as 
%                      terminal elements
%
%  Outputs:  phrbnd    vector of phrase boundary values before and 
%                      after the sentence and between syllables
%            phrtypes  a list of phrase types, one for each phrase boundary

  
punctcons = 'PUNCT';  % constituent for punctuations

% array defining the final mapping for the phrase boudries
phrbndmapping = [ 0 1 5 ];

% defines phrasing degree (default = 1)
degree = 1;

% defines the tree level of root node
rootlevel = 3;

fprintf('\nPhrasing\n============\nString: %s\n\n');
disptree(tree, 'sentence');
fprintf('\n');

%
% 1) INITIAL PHRASE BOUNDARIES
%
phrbnd(1) = 0;
phrtypes{1} = '';
j = 1;
wcons = find([tree(:).term]);
for i = 1:length(wcons)-1,
  if ~strcmp(tree(wcons(i)).name,punctcons),
    common = intersect(tree(wcons(i)).path, tree(wcons(i+1)).path);
    if isempty(common),
      fprintf('*** No common ancestor node:!\n');
      fprintf('%d',tree(wcons(i)).path); fprintf('\n');
      fprintf('%d',tree(wcons(i+1)).path); fprintf('\n');
      phrbnd(j+1) = rootlevel;
    else
      phrbnd(j+1) = max([tree(common).level]) + rootlevel;
    end
    phrtypes{j+1} = '';
    j = j+1;
  else 
    % set PUNCTUATIONS' PHRASE BOUNDARIES
    % Note: a punctuation is not a term and must therefore
    %       replace the previous boundry.
    phrbnd(j) = 1;
  end
end
phrbnd(j+1) = 0;
phrtypes{j+1} = '';

fprintf('After Initialisation:       ');
fprintf('%2d ',phrbnd); fprintf('\n');

%
% set SPECIAL CONSTITUENTS' PHRASE BOUNDARIES
%

% remove empty constituents from list of word constituents
wcons = find([tree(:).term]);
for i = 1:length(wcons),
  if isempty(tree(wcons(i)).sylnr),
    wcons(i) = 0;
  end
end
wcons = wcons(find(wcons>0));

for i = 1:length(wcons)-1,
  if strcmp(tree(wcons(i)).name,'V'),
    if phrbnd(i)>2,
      phrbnd(i) = 2;
    end
  end
  if strcmp(tree(wcons(i)).name,'CONJ'),
    if phrbnd(i)>2,
      phrbnd(i) = 2;
    end
  end
end

fprintf('After setting special cons: ');
fprintf('%2d ',phrbnd); fprintf('\n');


%
% 2) CYCLIC DELETION OF PHRASE BOUNDARIES
% 

acc = [tree(wcons).acc];
sylnr = [tree(wcons).sylnr];
if length(acc) ~= length(phrbnd)-1 | length(sylnr) ~= length(phrbnd)-1,
  fprintf('*** nr of word cons (%d) does not fit nr of phrbnds (%d)\n', ...
               length(acc),length(phrbnd));
  return
end

%
% first cyclic deletion (MELTING OF CLITIC WORDS)
%
fprintf('\nFirst cyclic deletion (melting of clitic words)\n\n');

pbmask = ones(1,length(phrbnd));
highest_bnd = max(phrbnd.*pbmask);
maxpb = find(phrbnd == highest_bnd);
while highest_bnd >= 2,
  for i = 1:length(maxpb),
    % next phrase boundary to left (>= 0)
    pbl = maxpb(i)-1;
    while ((pbl>0) & (phrbnd(pbl)<0)),
      pbl = pbl-1;
    end
    % next phrase boundary to right (>= 0)
    pbr = maxpb(i)+1;
    while ((pbr<length(phrbnd)) & (phrbnd(pbr)<0)),
      pbr = pbr+1;
    end
	 
    naccl = length(find(acc(pbl:maxpb(i)-1)>0));
    naccr = length(find(acc(maxpb(i):pbr-1)>0));

    if (phrbnd(maxpb(i))>=phrbnd(pbl)) & (naccl==0),
      phrbnd(maxpb(i)) = -1;
    end
    if (phrbnd(maxpb(i))>phrbnd(pbr)) & (naccr==0),
      phrbnd(maxpb(i)) = -1;
    end
  end
  % mask boundaries already used
  pbmask(maxpb) = 0;
  
  fprintf('After checking %2der bounds: ', highest_bnd); 
  fprintf('%2d ',phrbnd); fprintf('\n');

  highest_bnd = max(phrbnd.*pbmask);
  maxpb = find(phrbnd == highest_bnd);
end

%
% second cyclic deletion (RHYTHMIC MELTING OF TEMPORARY PHRASES)
%
fprintf('\nSecond cyclic deletion (rhythmic melting of temporary phrases)\n\n');

pbmask = ones(1,length(phrbnd));
highest_bnd = max(phrbnd.*pbmask);
maxpb = find(phrbnd == highest_bnd);
while highest_bnd >= 2,
  for i = 1:length(maxpb),
    % next phrase boundary to left (>= 0)
    pbl = maxpb(i)-1;
    while ((pbl>0) & (phrbnd(pbl)<0)),
      pbl = pbl-1;
    end
    % next phrase boundary to right (>= 0)
    pbr = maxpb(i)+1;
    while ((pbr<length(phrbnd)) & (phrbnd(pbr)<0)),
      pbr = pbr+1;
    end
	 
    naccl = length(find(acc(pbl:maxpb(i)-1)>0));
    naccr = length(find(acc(maxpb(i):pbr-1)>0));

    nsyll = sum(sylnr(pbl:maxpb(i)-1));
    nsylr = sum(sylnr(maxpb(i):pbr-1));
	 
    q = degree;
    if (nsyll<=2), 
      q = degree+1;
    elseif (nsyll>=4),
      q = degree-1;
    end
    if (((phrbnd(maxpb(i))>2) | (degree>=2)) & ...
        (phrbnd(maxpb(i))>=phrbnd(pbl)) & ...
        (phrbnd(maxpb(i))>=phrbnd(pbr)) & (naccl <= q)),
      phrbnd(maxpb(i)) = -1;
    end
    q = degree;
    if (nsylr<=2), 
      q = degree+1;
    elseif (nsylr>=4),
      q = degree-1;
    end
    if (((phrbnd(maxpb(i))>2) | (degree>=2)) & ...
        (phrbnd(maxpb(i))>=phrbnd(pbl)) & ...
        (phrbnd(maxpb(i))>phrbnd(pbr)) & (naccr <= q)),
      phrbnd(maxpb(i)) = -1;
    end
  end
  % mask boundaries already used
  pbmask(maxpb) = 0;
  
  fprintf('After checking %2der bounds: ', highest_bnd); 
  fprintf('%2d ',phrbnd); fprintf('\n');

  highest_bnd = max(phrbnd.*pbmask);
  maxpb = find(phrbnd == highest_bnd);
end


%
% 3) NORMALIZATION OF PHRASE BOUNDARIES
% 

for i = 1:length(phrbnd)
  normbnd = find(phrbndmapping >= phrbnd(i), 1);
  if isempty(normbnd) || (phrbnd(i) < 0)
    phrbnd(i) = -1;
  else
    phrbnd(i) = normbnd - 1;
  end
end

fprintf('\n\nAfter normalization:        '); 
fprintf('%2d ',phrbnd); fprintf('\n');

%
% 4) SET PHRASE TYPES
%
phrtypes{1} = 'P';
for i = 1:length(phrbnd),
  if phrbnd(i) >= 0,
    phrtypes{i} = 'P'; % set all phrases to progredient phrase type
  end
end
phrtypes{end} = 'E'; % last phrase boundary is specially marked

% set last phrase as terminal phrase type
i = length(phrtypes)-1;
while (i>1) & strcmp(phrtypes{i},''),
  i = i-1;
end
phrtypes{i} = 'T';

