function tree = accentuation_sol(tree);
%
%  tree = accentuation(tree);
% 
%  Assignment of sentence stress positions and values. 
%
%  Input:   tree   annotated syntax tree with words as 
%                  terminal elements
%
%  Output:  tree   modified annotated syntax tree
 

%
% CONFIGURATION PARAMETERS
%

% List of unaccentated constituents
nonacclst = {'ART'; 'PRAEP'; 'PRON'; 'REFL'; 'CONJ'; 'PUNCT'; 'PERS'};

% list of accentuation patterns
accpat(1).cons = 'NG';
accpat(1).acc = 'end';

accpat(2).cons = 'VG';
accpat(2).acc = 'end';

accpat(3).cons = 'ADVGN';
accpat(3).acc = 'begin';

accpat(4).cons = 'S';
accpat(4).acc = 'end';

accpat(5).cons = '*';
accpat(5).acc = 'end';

% list of rhythmic stress shift patterns
accshiftpat(1).old = [3 2 1];
accshiftpat(1).new = [2 3 1];

accshiftpat(2).old = [2 2 1];
accshiftpat(2).new = [2 3 1];

accshiftpat(3).old = [1 2 3];
accshiftpat(3).new = [1 3 2];

accshiftpat(4).old = [1 2 2];
accshiftpat(4).new = [1 3 2];

% list of constituents that are reduced after cyclic accentuation 
reducepat{1} = 'V';

fprintf('Accentuation of string: %s\n\n');
print_subtree_graph(tree, 1);
fprintf('\n\n');

%
% 1) INITIAL SENTENCE ACCENTS
%

% go through all terminal nodes of the tree
for i = find([tree(:).term]),
  for j = 1:length(nonacclst),
          % unaccented constituents are set to accentuation -1
    if strcmp(tree(i).name,nonacclst(j)),
      tree(i).acc = -1;
    end
  end
end

fprintf('After Initialization:        ');
print_accentuation(tree); 

%
% 2) CYCLIC ACCENTUATION RULES
%

% save the level of each node as a matrix (just for readability)
treelevels = [tree(:).level];
% dto. for terminal nodes
termnodes = [tree(:).term];

% walk each node of the tree starting at the leaves and ending at the root
for level = max(treelevels):-1:0
  for nodeid = find((treelevels == level) & ~termnodes)

		% NUCLEAR STRESS RULE

    % search through the accentuation patterns
    patid = 1;
    while ~(strcmp('*',accpat(patid).cons) || ...
            strcmp(tree(nodeid).name,accpat(patid).cons))
      patid = patid + 1;
    end
    % accentuation pattern that applies, one of:
    %  'begin' - nucleus on first "1" accent
    %  'end'   - nucleus on last "1" accent
    acc_pattern = accpat(patid).acc;

    % Find the nodes where the nuclear stress rule needs to be
    % applied: the terminals of the subtree of current constituent 'nodeid'.
    nsr_nodes = [];
    for i=1:length(tree),
      if ~isempty(tree(i).sylnr) && ismember(nodeid, tree(i).path)
        nsr_nodes = [ nsr_nodes i];
      end
    end

    % save the accents of the interesting nodes in in a vector
    nsr_accents = [ tree(nsr_nodes).acc ];

    fprintf('>> Substring: ');
    print_subtree_graph(tree, nodeid);
    fprintf('\n');

% >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

    % find all "1" accents
    first_accents = find(nsr_accents == 1);

    % Reduction is only necessary if there is more than one "1" accent
    if length(first_accents) > 1,
    % find the nucleus node
      if strcmp(acc_pattern, 'begin')
        nucleus = first_accents(1);
      else
        nucleus = first_accents(end);
      end
      % reduce all accents apart from the nucleus
      for n = 1:length(nsr_nodes)
        if (nsr_accents(n) >= 0) && (n ~= nucleus)
          tree(nsr_nodes(n)).acc = tree(nsr_nodes(n)).acc + 1;
        end
      end
    end

% >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

    fprintf('\tAfter nuclear stress:        ');
    print_accentuation(tree);

    % rhythmic stress shift rule
    tree = rhythmicstressshift(tree, nodeid, accshiftpat);
    fprintf('\tAfter rhythmic stress shift: ');
    print_accentuation(tree);
  end
end

%
% 3) POSTCYCLIC ACCENTUATION RULES
%

% postcyclic accent reduction of certain constituents, e.g. V

for i = find([tree(:).term]),
  for p = 1:length(reducepat),
    if strcmp(tree(i).name,reducepat{p}) & (tree(i).acc > 1),
      tree(i).acc = tree(i).acc + 1;
      break
    end
  end
end
fprintf('\nAfter postcyclic reduction:  ');
print_accentuation(tree); 
end



%--------------------------------------------------------------------
%
% HELPER OUTPUT FUNCTIONS
%

function print_accentuation(tree)
% Print accentuation pattern of tree
  for i = 1:length(tree),
    if tree(i).sylnr > 0,
      fprintf('%d ',tree(i).acc);
    end
  end
  fprintf('\n');
end


function print_subtree_graph(tree,nodeid)
% Print the graphemic representation of a subtree of tree
  for i = 1:length(tree),
    if ~isempty(tree(i).graph) && ismember(nodeid, tree(i).path)
      fprintf('%s', [regexprep(tree(i).graph, '[+$#]', '') ' ']);
    end
  end
end
