function  tts_synthesis(inp,vers,fign);
%
%  function  tts_synthesis(inp,vers,fign);
%
%  Text-to-speech synthesis:  the text string `inp` is transformed into 
%  a speech signal that can be played interactively.
%
%  Inputs:   inp   either the name of a file that contains the text
%                  or directly the input string
%            vers  optional argument that allows to select among different 
%                  versions of accentuation and phrasing functions;
%                  for `vers` = 'xyz' the functions accentuation_xyz.m 
%                  and phrasing_xyz.m are used if they exist
%            fign  use figure window `fign` [default: 1]

if nargin < 3 
  fign = 1;
end
  
if (nargin < 2) || isempty(vers)
  accentuation_funct_name = 'accentuation';
  phrasing_funct_name = 'phrasing';
else
  accentuation_funct_name = ['accentuation_' vers];
  phrasing_funct_name = ['phrasing_' vers];
end
                % check which accentuation function can be used
if ~exist(accentuation_funct_name)   
  nams = {'accentuation' 'accentuation_frame' 'accentuation_sol'}; 
  accentuation_funct_name = get_other_func(accentuation_funct_name,nams);
  if isempty(accentuation_funct_name)
    disp('(program aborted)')
    return
  end
end
                % check which phrasing function can be used
if ~exist(phrasing_funct_name) 
  nams = {'phrasing' 'phrasing_frame' 'phrasing_sol'}; 
  phrasing_funct_name = get_other_func(phrasing_funct_name,nams);
  if isempty(phrasing_funct_name)
    disp('(program aborted)')
    return
  end
end

  
% setup graphics window
figs = get(0,'children');
if isempty(figs) | all(figs ~= fign) | ~strcmp(get(fign,'tag'),'TTS')
  figure(fign)
  clf
  pos = [100 100 1200 800];
  set(gcf,'position',pos);
  set(gcf,'tag','TTS');
end 
  
figure(fign)
clf
set(gcf,'visible','off');   % avoid that the command window gets
drawnow;                    % occluded by the figure window 

h1 = subplot('Position',[0.05 0.52 0.9 0.43]);  % for syntax tree
h2 = subplot('Position',[0.05 0.25 0.9 0.17]);  % for F0 contour
h3 = subplot('Position',[0.05 0.05 0.9 0.17]);  % for speech signal
                                                % for phonological description
ht = uicontrol(gcf, 'Tag','text_ptr', 'Style','text', 'Enable','on', ...
	  'Units','normalized', 'FontSize',12, ...
	  'BackgroundColor',[0.85 0.85 0.85], 'ForegroundColor',[0 0 0], ...
	  'Position',[0.05 0.43 0.9 0.06], 'HorizontalAlignment','center', ...
	  'String','');


% morphological and syntactic analysis of the sentence 
% ----------------------------------------------------
subplot(h1);
tree = morphosyntactic_analysis(inp);


% sentence accentuation
% ---------------------
eval(['tree = ' accentuation_funct_name '(tree);']);

% sentence phrasing
% -----------------
eval(['[phrbnd, phrtypes] = ' phrasing_funct_name '(tree);']);

% sentence accent normalization
% -----------------------------
tree = accentnormalization(tree, phrbnd);

% phonological representation
% ---------------------------
[pholrep, ptrtxt] = build_phonolrepr(tree,phrbnd,phrtypes);
set(ht,'String',ptrtxt);

% generation of fundamental frequency contour and phone durations
% ---------------------------------------------------------------
subplot(h2);
[elems, el_durs, el_freqs] = prosody_generation(pholrep);

% synthesizing of the speech signal based on LPC-concatenation of 
% diphone elements
% -----------------------------------------------------------------------
subplot(h3);
[sig, SampleFreq] = signal_generation(elems, el_durs, el_freqs);

% all went well, so show the graphics
set(gcf,'visible', 'on');
drawnow;

% finally play the synthesized signal
% -----------------------------------------------------------------
reply = 'y';
while ~strcmpi(reply,'n'),
  sound(sig, SampleFreq);
  reply = input('Play again? (Y/n):  >> ','s');
end


%-----------------------------------------------------------------------
%  helper function
%-----------------------------------------------------------------------

function  fnam = get_other_func(xnam,nams)
%  get another function out of `nams` which is different 
%  from `xnam`

idx = strcmp(xnam,nams);  % remove `xnam` from `nams`
nams = nams(idx==0);

disp(['function not found:  ' xnam])
fnam = '';
for i = 1:length(nams)
  if exist(nams{i}) == 2
    ans = input(['do you want to use  ' nams{i} '  instead?  (Y/n)  >> '],'s');
    if isempty(ans) || ans ~= 'n'
      fnam = nams{i};
    end
  end  
  if ~isempty(fnam)  
    break
  end
end

