function [elems, el_durs, el_freqs] = prosody_generation (pholrep);
%  
%  [elems, el_durs, el_freqs] = prosody_generation (pholrep);
%
%  Estimation of phone duration and fundamental frequency.
%
%  Uses the following prosody models (default):
%    get_dur_rule_conditions.m, DurModel.mat and
%    get_F0_rule_conditions.m, F0Model.mat
%  if these files exist, or otherwise the solutions of the prosody 
%  excercises 7 and 8, i.e.
%    get_dur_rule_conditions_sol.m, DurModel2.mat
%    get_F0_rule_conditions_sol.m, F0Model2.mat
%
%  Input:
%    pholrep - phonetic representation of the sentence
%              struct of arrays that contains for each syllable:
%              .syl   phonologic string
%              .acc   accentuation strength
%              .phr   phrase type of surrounding phrase ('P' or 'T')
%              .lng   language of syllable (unused, always '-')
%              .bnd   boundary after syllable, one of:
%                     '-'       syllable boundary
%                     '|'       word boundary
%                     '0' - '3' phrase boundary
%
%  Output:
%   elems    - array of half diphone elements to use
%   el_durs  - duration of each element
%   el_freqs - fundamental frequency of each element



%   Generiere Lautsequenz mit linguistischen Faktoren
%-------------------------------------------------------
%
S = load_phonol_repr(pholrep.syl,pholrep.acc,pholrep.lng,...
                        pholrep.phr,pholrep.bnd);


%   Grundfrequenzmodell laden und Grundfrequnezwerte schaetzen
%------------------------------------------------------------
%
if exist('F0Model1.mat') & exist('get_F0_rule_conditions.m')
  load F0Model1  
  C = get_F0_rule_conditions(S);
elseif exist('F0Model2.mat') & exist('get_F0_rule_conditions_sol.m')
  load F0Model2  
  C = get_F0_rule_conditions_sol(S);
else
  if ~exist('F0Model2.mat')
    error('*** file not found:  F0Model2.mat')
  else
    error('*** file not found:  get_F0_rule_conditions_sol.m')
  end
end
m1.F0 = C*F0mod.q;


%   Dauermodell laden und Dauerwerte schaetzen
%--------------------------------------------
%
if exist('DurModel1.mat') & exist('get_dur_rule_conditions.m')
  load DurModel1  
  C = get_dur_rule_conditions(S);
elseif exist('DurModel2.mat') & exist('get_dur_rule_conditions_sol.m')
  load DurModel2
  C = get_dur_rule_conditions_sol(S);
else
  if ~exist('DurModel2.mat')
    error('*** file not found:  DurModel2.mat')
  else
    error('*** file not found:  get_dur_rule_conditions_sol.m')
  end
end
m1.dur = 10.^(C*Durmod.q);


%
%   Erzeugen der Semidiphonsequenz
%------------------------------------
%
[sd sdi] = get_semidiphones(S.phone_id);


%   Lautdauersequenz 
%   auf Semidiphonsequenz uebertragen
%----------------------------------------------
%
sdd = zeros(length(sdi),1);
for i = 1:size(m1.dur,1)
  phi = find(sdi == i);
  l = zeros(length(phi),1);
  for j = 1:length(phi)
	 sig = get_corp_elem(sd{phi(j)},0);
	 l(j) = length(sig);
	 clear sig
  end
  for j = 1:length(phi),
	 sdd(phi(j)) = m1.dur(i)*l(j)/sum(l);
  end
end 


%   Grundfrequenzkontur 
%   auf Semidiphonsequenz uebertragen
%----------------------------------------------
%
t = zeros(size(m1.dur,1),1);
f = t; 
j = 1;
for i = 1:size(S.phone_id,1)
  t(i) = round(sum(m1.dur(1:i)) - m1.dur(i)/2)/1000;
  if phone_has_properties(S.phone_id(i), 'nucleus'), 
	 f(i) = m1.F0(j);
	 j = j+1;
  else
	 f(i) = 0;
  end
end
f = interp1(t(f>0),f(f>0),t,'linear',NaN);
% set F0 values of first onset and last offset. 
for i = 2:size(t,1)
  if isnan(f(i))
	 f(i) = f(i-1);
  end
end
for i = size(t,1)-1:-1:1
  if isnan(f(i))
	 f(i) = f(i+1);
  end
end


sdF0 = zeros(length(sdd),1); 
for i = 1:size(m1.dur,1)
  phi = find(sdi == i);
  sdF0(phi) = f(i);
end


%   Semidiphonsequenz anzeigen
%---------------------------------
%
for i = 1:size(sd,1)
  fprintf('%s \t%d \t%.2f \t%.2f\n',sd{i},sdi(i),sdd(i),sdF0(i));
end

syls = find(S.phone_number == 1);
syle = [syls(2:end)-1; size(S.phone_number,1)];

sylbnd = zeros(length(syls)+1,1);
for i=1:length(syls),
  sylbnd(i+1) = sylbnd(i)+sum(m1.dur(syls(i):syle(i)));
end

plot(t,f);
xlim = [0 sylbnd(end)/1000];
ylim = [0.9*min(f) 1.1*max(f)];
set(gca,'xlim',xlim)
set(gca,'ylim',ylim)
line([sylbnd(2:end-1)'/1000; sylbnd(2:end-1)'/1000],...
	  [0.9*min(f) 1.1*max(f)], ...
	  'LineStyle','--', 'Color',[0.8 0.8 0.8]);
for i = 2:size(sylbnd,1),
  syltxt = regexprep(deblank(pholrep.syl(i-1,:)),'_','\\_');
  text((sylbnd(i-1)+sylbnd(i))/2/1000,0.95*min(f),syltxt,...
		 'HorizontalAlignment','center');
end
drawnow


elems    = char(sd);
el_durs  = sdd(:);
el_freqs = sdF0(:);
