function tree = morphosyntactic_analysis(txtinput);
%  
%  tree = morphosyntactic_analysis(txtinput);
%
%  Morpho-syntactic analysis of an arbitrary text. The text is parsed
%  and saved in a tree structure. In addition, the tree is printed in
%  the currently active figure.
%
%  It uses a grammar called 'ueb18.dat' and a finite-state transducer 
%  (two-level rules that have been converted into an automaton) called
%  'mophon.aut'. The original automaton does German 'Auslautverhaertung'
%  and syllabification.
%
%  Example input can be found in the files 'satz01.txt' ... 'satz10.txt'.
%
%  Input:
%    txtinput  - either the name of a file that contains the text
%                or directly the input string.
%
%  Output:
%    tree      - annotated tree, an array of node structures with
%                the folowing fields:
%                 .id    - node id (same as index into array)
%                 .path  - array describing path to root node
%                 .name  - name of constituent
%                 .level - depth of node within the tree
%                 .term  - 1 if the node is a terminal node
%                 .graph - graphemic string (for word constituents)
%                 .phon  - phonemic string (for word constituents)
%                 .sylnr - number of syllables (for word constituents)
%                 .accnr - number of stressed syllables
%                 .acc   - strength of accentuation of main accent
  
tree = [];

% check if the given input is a file name
txtfln = [];
if exist(txtinput, 'file')
  txtfln = txtinput;
elseif exist(['saetze/' txtinput], 'file')
  txtfln = ['saetze/' txtinput];
elseif exist(['Gegebenes/saetze/' txtinput], 'file')
  txtfln = ['Gegebenes/saetze/' txtinput];
end

if ~isempty(txtfln)
  txtinput = char(textread(txtfln, '%[^\n]', 1));
end

gramfil = 'ueb18.dat';
if ~exist(gramfil,'file')
  error(['*** grammar file not found:  ' gramfil])
end

% call the external parser to perform a morpho-syntactic analysis
[c,res] = parse_text(gramfil,txtinput);

delim = char('=' * ones(1,74));
fprintf('Parser output: \n%s\n%s\n%s\n\n', delim, res, delim);
if c ~= 0
  return
end

% find the bracket notation of the first solution starting with "TREE:"
pos = strfind(res, 'TREE:');
if isempty(pos)
  error('*** keyword "TREE:" not found in parser output');
end
idx = regexp(res(pos:end),'\n');   % get the positions of the linebreaks
if length(idx) < 2
  error(['*** parser output seems not to contain the syntax tree' ...
         ' in bracket notation']);
end

% convert it into the tree structure
aststring = res(pos+idx(1):pos+idx(2)-1);
tree = read_syntphol(aststring);

% show the tree in the current figure
disptree(tree);
