function [repr, txt] = build_phonolrepr (tree, phrbnd, phrtypes);
  
% [repr, txt] = build_phonolrepr (tree, phrbnd, phrtypes)
%
% Build phonological representation from annotated syntax tree 'tree' and
% phrase boundaries in 'phrbnd' and 'phrtypes'.
%
% Input:
%   tree      annotated syntactic tree incl. phonetic strings of
%                    word constituents.
%   phrbnd    list of phrase boundary values (nr of word cons + 1).
%   phrtypes  list of phrase types.
%  
% Outputs: 
%   repr      Structure of phonological representation for each syllable
%       .syl   phonologic string
%       .acc   accentuation strength
%       .phr   phrase type of surrounding phrase ('P' or 'T')
%       .lng   language of syllable (unused, always '-')
%       .bnd   boundary after syllable, one of:
%               '-'       syllable boundary
%               '|'       word boundary
%               '0' - '3' phrase boundary

if (length([tree.sylnr]) ~= (length(phrbnd)-1)),
  error('*** Unbalanced word cons (%d) and phrase boundaries (%d)\n', ...
			 length([tree.sylnr]), length(phrbnd));
end

% Build structure representation

repr.syl = [];
repr.acc = [];
repr.phr = [];
repr.lng = [];
repr.bnd = [];

% current position in phrase boudary arrays
phrpos = 1;
% current phrase type
curphrtype = phrtypes{1};
for nid = 1:length(tree)
	if ~isempty(tree(nid).phon),
		% add the sylable content and accents
		sylstart = [ 1 strfind(tree(nid).phon, '-') + 1 length(tree(nid).phon) + 2 ];
		for i = 1:length(sylstart)-1
			start = sylstart(i);
			% set the accent
			if isempty(tree(nid).acc)
				repr.acc = strvcat(repr.acc, '0');
			elseif strcmp(tree(nid).phon(start), char(39))
				repr.acc = strvcat(repr.acc, num2str(tree(nid).acc));
				start = start + 1;
			elseif strcmp(tree(nid).phon(start), ',')
				repr.acc = strvcat(repr.acc, '4');
				start = start + 1;
			else
				repr.acc = strvcat(repr.acc, '0');
			end
			% set the syllable content (without -,')	
			repr.syl = strvcat(repr.syl, tree(nid).phon(start:sylstart(i+1)-2));
			% language is ignored
			repr.lng = [ repr.lng; '-' ];
			% phrase boundry type
			repr.phr = [ repr.phr; curphrtype ];
			% phrase boundry strength, preliminarily a end of syllable
			repr.bnd = [repr.bnd; '-'];
		end
		% correct the final phrase boundary
		phrpos = phrpos + 1;
		if phrbnd(phrpos) < 0
			repr.bnd(end, :) = '|';
		else
			repr.bnd(end, :) = sprintf('%d',phrbnd(phrpos));
		end
		% jump over empty phrase type descriptions
		if ~isempty(phrtypes{phrpos}) && ~strcmp(phrtypes{phrpos}, 'E')
			curphrtype = phrtypes{phrpos};
		end
	end
end


% Build textual representation

txt = '';
j = 1;
for i=1:length(tree),
  if ~isempty(tree(i).phon),
	 if ~isempty(tree(i).acc),
		if phrbnd(j)>=0,
		  txt = [txt sprintf('#{%d} ',phrbnd(j))];
		  txt = [txt sprintf('(%s) ',phrtypes{j})];
		end
		j = j+1;
		if (tree(i).acc>0),
		  tree(i).phon = strrep(tree(i).phon,char(39), ...
										sprintf('[%d]',tree(i).acc));
		end
	 end
	 tree(i).phon = strrep(tree(i).phon,char(39),'');
	 tree(i).phon = strrep(tree(i).phon,',','[4]');
	 txt = [txt sprintf('%s- ',tree(i).phon)];
  end
end

% remove last '-'
if txt(end-1) == '-',
  txt(end-1) = '';
end
