function  ok = record_word(filnam,dirnam,tstr)
%
%  function  ok = record_word(filnam,dirnam,tstr)
%
%  Generates a simple user interface for interactivly recording
%  a word and storing it in wave format on file
%
%  Input:   filnam   file in which the recorded signal is to be stored;
%                    if empty, the file name has to be entered interactively
%           dirnam   name of an existing subdirectory of the current
%                    directory; if empty, the signal will be stored in 
%                    the current directory
%           tstr     optional string to be displayed as GUI title 
%
%  Output:  ok       completion flag:  1 = success
%                                      0 = failure

%----- define global variable -----

global rec_handle; 


%----- set default values of parameters -----

fs_adc = 16000;  % necessary to prevent aliasing of some USB headsets
fs = 8000;
dur = 3;         % recording time in seconds
delta = 0.05;
winlen = 240;
winshf = 80;

if isempty(filnam)
  filnam = 0;
  autoclose = 0;
else
  autoclose = 1;
end
if isempty(dirnam)
  path_name = [''];
else
  path_name = [dirnam '/'];
  if ~exist(path_name,'dir')
    error(['*** no such directory:  ' dirnam])  
  end
end


%----- GUI definition: set plot area and UIcontrols in figure `fig` -----

fig = 3;            % GUI appears as figure `fig`
hf = figure(fig);
clf
hs = axes('position',[0.1 0.5 0.8 0.4]);
if exist('tstr','var')
  title(tstr,'fontsize',16,'interpreter','none')
end
tx = 0.5;
ty = 0.5;

h_start_button = uicontrol('style','pushbutton','string','RECORD', ...
           'units','norm','position',[0.05 0.05 0.17 0.08], ...
           'userdata',0, ...
           'callback','set(findobj(''string'',''RECORD''),''userdata'',1)');

h_stop_button = uicontrol('style','pushbutton','string','PLAY', ...
           'units','norm','position',[0.25 0.05 0.17 0.08], ...
           'userdata',0, ...
           'callback','set(findobj(''string'',''PLAY''),''userdata'',1)');

h_stop_button = uicontrol('style','pushbutton','string','STORE', ...
           'units','norm','position',[0.45 0.05 0.17 0.08], ...
           'userdata',0, ...
           'callback','set(findobj(''string'',''STORE''),''userdata'',1)');

h_close_button = uicontrol('style','pushbutton','string','CLOSE', ...
           'units','norm','position',[0.78 0.05 0.17 0.08], ...
           'userdata',0, ...
           'callback','set(findobj(''string'',''CLOSE''),''userdata'',1)');

h_msg = uicontrol('style','text','units','norm','position',[0.1 0.2 0.8 0.15], ...
        'string','');


%----- check if USB headset is connected and get its input ID -----

try 
  vstr = version;
  if str2num(vstr(1:2)) >= 9
    in_dev_id = audiodevinfo(1,'USB');
  else
    au_dev_info = audiodevinfo;  % patch for a bug in older Matlab versions
    in_dev_id = -1;
    for i = 1:length(au_dev_info.input)
       if regexp(au_dev_info.input(i).Name,'USB')
         in_dev_id = audiodevinfo(1,au_dev_info.input(i).Name);
       end
    end
  end
  in_dev_name = audiodevinfo(1,in_dev_id);
  USB_ok = 1;
catch
  % disp(['No USB headset found. Connect USB headset and restart Matlab!'])
  % return
  USB_ok = 0;
  %  set(h_msg,'string',['Recording and play-back will be done via ' ...
  %                    'default device']);
end
str{1} = 'Recording and play-back will be done via';
if USB_ok 
  str{2} = in_dev_name;
else
  str{2} = 'the default device';
end
set(h_msg,'string',str);


%----- handle recorder ----- 

if ~isempty(rec_handle)
  delete(rec_handle);    % delete existent recorder
end


%----- record and store signals -----

sig = [];
if nargout > 0 
  ok = 0;
end
                                     % repeat until "CLOSE" button pressed
while get(findobj('string','CLOSE'),'userdata') == 0

                                       % wait until "RECORD" button is pressed
  while get(findobj('string','RECORD'),'userdata') == 0
                                         % test if "CLOSE" button pressed
    if get(findobj('string','CLOSE'),'userdata') == 1
      close(fig)
      return
    end
                                         % test if "PLAY" button pressed
    if get(findobj('string','PLAY'),'userdata') == 1
      if ~isempty(sig)
        if usejava('jvm')                % if Java activated
          sound(sig,fs);
        else
          sound(sig);
        end
      end
      set(findobj('string','PLAY'),'userdata',0)
    end

                                         % test if "STORE" button pressed
    if get(findobj('string','STORE'),'userdata') == 1
      if ~isempty(sig)
        if isempty(fr1)
          h = text(tx,ty,'No utterance found in this signal!', ...
                   'fontsize',20,'horiz','center');
          pause(2)
          delete(h)
          return
        end
        if filnam == 0
          fln_filter = fullfile(path_name, '*.wav')
          [filnam path_nam] = uiputfile([path_name '/*.wav'], ...
                                      'Save Signal in File ...');
        end
        if filnam == 0,              % if cancel button pressed or error
          h = text(tx,ty,'Signal not stored!','fontsize',20,'horiz','center');
          pause(2)
          delete(h)
        else
          if isempty(findstr(filnam,'.wav'))
            filnam = [filnam '.wav'];
          end
          audiowrite([path_name filnam],sig,fs,'BitsPerSample',16);
          filnam = 0;
          if nargout == 1
            ok = 1;
          end
          if autoclose == 1      % return if signal stored on desired file 
            close(hf)
            return
          end
        end
      end
      set(findobj('string','STORE'),'userdata',0)
    end
    pause(0.1)
  end

                        % record signal of length `dur` 
  nbits = 16;
  nchan = 1;
  if USB_ok
    rec_handle = audiorecorder(fs_adc,nbits,nchan,in_dev_id);
  else
    rec_handle = audiorecorder(fs_adc,nbits,nchan);
  end
  record(rec_handle);
  h = text(tx,ty,'Please speak now!','fontsize',20,'horiz','center');
  drawnow
  pause(dur);
  delete(h)
  drawnow
  stop(rec_handle);
  sig_x = getaudiodata(rec_handle);
  sig_a = resample(sig_x,fs,fs_adc);  
  
  if length(sig_a) < dur*fs
    warning(['*** only ' num2str(n) ' samples read instead of ' ...
            num2str(dur*fs)]) 
  end

                             % remove low frequencies (rumble filter)
  A = [1.0000 -2.8123  2.6394 -0.8261];
  B = [0.9099 -2.7290  2.7290 -0.9099];
  if ~exist('A','var') | isempty(A)
    hp_ord = 3;                 % design filter if polynominal not given
    fr_cut = 120;
    pass_ripp = 0.5;
    stop_attn = 40;
    [B,A] = ellip(hp_ord,pass_ripp,stop_attn,fr_cut/fs*2,'high');
  end

  sig = filter(B,A,sig_a);
  sig(sig >  0.9999) =  0.9999;            % limit signal: -1...+1
  sig(sig < -0.9999) = -0.9999;
                                           % detect endpoints 
  numfra = floor((length(sig)-(winlen-winshf))/winshf);
  rms = zeros(numfra,1);
  for nfr = 1:numfra;
    ns = (nfr-1)*winshf;
    s = sig(ns+1:ns+winlen);                  
    rms(nfr) = sqrt(sum(s.*s)/winlen);
  end 
  [fr1,fr2] = detect_endpoints(rms,1);

  figure(hf);                              % plot the signal 
  t = (1:length(sig))/fs; 
  plot(t,sig)
  set(gca,'xlim',[0 t(end)])
  ylims = get(gca,'ylim');
  ylims = [min(ylims(1),-0.1) max(ylims(2),0.1)];
  set(gca,'ylim',ylims);
  tx = t(end)/2;
  ty = ylims(2)-(ylims(2)-ylims(1))/4;
  nn = length(sig);
  n1 = round(fr1*winshf-delta*fs);
  n2 = round(fr2*winshf+delta*fs);

  if isempty(fr1)                          % no starting point 
    h = text(tx,ty,'No endpoints of utterance found!','fontsize',20, ...
          'horiz','center');
    pause(2)
    delete(h)
  elseif n1 < 1 
    h = text(tx,ty,'Too early spoken!','fontsize',20, ...
          'horiz','center');
    pause(2)
    delete(h)
  elseif n2 > nn 
    h = text(tx,ty,'Too late spoken!','fontsize',20, ...
          'horiz','center');
    pause(2)
    delete(h)
  else
    sig = sig(n1:n2);          % plot the extracted segment of the signal
    t = (1:length(sig))/fs;
    plot(t,sig)
    set(gca,'xlim',[0 t(end)])
    set(gca,'ylim',ylims);
    tx = t(end)/2;
    ty = ylims(2)-(ylims(2)-ylims(1))/4;
    hold on
    plot([1 1]*delta,ylims,':r')
    plot([1 1]*(t(end)-delta),ylims,':g')
    hold off
  end
  if exist('tstr','var')
    title(tstr,'fontsize',16)
  end
  set(findobj('string','RECORD'),'userdata',0)
end
