function [cep,rms] = extr_mfcc_and_rms(x,fs,nfilt,w,shift,ncep,nfft)
% 
%  function:  [cep,rms] = extr_mfcc_and_rms(x,fs,nfilt,w,shift,ncep,nfft)
%
%  Evaluate MFCC contour and intensity contour of speech signal.
%  Note that the 0th cepstral coefficient is not included and 
%  therefore the contour is independent of the signal amplitude.
%
%  Inputs:   x      input signal
%            fs     sampling frequency
%            nfilt  number of filters in mel filter bank
%            w      window function (discrete)
%            shift  offset in number of samples between consecutive 
%                     analysis windows 
%            ncep   order of cepstral vector
%            nfft   FFT resolution (optional argument, default = length(w))
%
%  Outputs:  cep    array of MFCC vectors (without zero quefrency component)
%            rms    vector of signal power (RMS) values

x = x(:);          % make column vector

if nargin < 7
  nfft = length(w);
end

siglen = length(x);
winlen = length(w);

if siglen < winlen 
  error('*** length of signal shorter than analysis window');
end
if (winlen < 1) | (max(abs(w)) == 0)
  error('*** invalid analysis window (zero size or all zeros)');
end

numfra = floor((siglen-(winlen-shift))/shift);
cep = zeros(numfra,ncep);
rms = zeros(numfra,1);

mfbk = [];      % if empty the filter bank will be defined in function mfcc
 
for nfr = 1:numfra;
  ns = (nfr-1)*shift;
  s = x(ns+1:ns+winlen);                  
  sw = s.*w;               % windowing
  [cepx,mfbk] = mfcc(sw,fs,nfilt,ncep+1,mfbk,0,0.5,nfft);
  cep(nfr,1:ncep) = cepx(2:ncep+1)';
  rms(nfr) = sqrt(sum(s.*s)/winlen);      % eval signal intensity (RMS) 
end
