function [dst,warpcurve,distcurve] = dtw(ref,tst,refwgt,tstwgt)
%
%  function: [dst,warpcurve,distcurve] = dtw(ref,tst,refwgt,tstwgt)
%
%  Dynamic time warping: The path extensions as shown in Figure 11.4c 
%  of the book "Sprachverarbeitung" are used.
%
%  Inputs:   ref        reference contour: sequence of feature vectors 
%                       of the reference template, where ref(n,:) 
%                       denotes feature vector n
%            tst        test contour (same format as ref)
%            refwgt     optional weight vector for the refenence contour:
%                       if present, the local distances are weighted with
%                       `refwgt(ix)+tstwgt(iy)`
%            tstwgt     optional weight vector for the test contour
%
%  Outputs:  dst        mean distance along the warp curve
%            warpcurve  warp curve (matrix with indices of feature vectors
%                       of reference contour (1st column) and test
%                       contour (2nd column))
%            distcurve  local distances along the warp curve


% macros for weighting of the local distances:
% a) loc weight is:  1  (i.e. no weighting)
     locweight_type_0 = 'ones(sx,sy)';   
% b) loc weight is:  refwgt(ix)+tstwgt(iy)
     locweight_type_1 = 'refwgt*ones(1,sy)+ones(sx,1)*tstwgt'''; 

locweight_warp = locweight_type_1;   % weight for time warping 
locweight_dist = locweight_type_1;   % weight for dist computation 


%----- check the input arguments -----

if (nargin < 2) | (nargin == 3)
  error('*** wrong number of arguments');
end
 
[sx ord] = size(ref);
[sy aux] = size(tst);
if ord ~= aux 
  error('*** order of the ref and test contour vectors must be equal')
end
if (sx >= 2*sy) | (sx*2 <= sy)
  disp('warning: length of patterns differ more than a factor of 2')
  dst = 999;
  warpcurve = [];
  distcurve = [];
  return;
end

if nargin == 2
  locweight_warp = locweight_type_0;
  locweight_dist = locweight_type_0;
else
  [i j] = size(refwgt);
  if i ~= sx
    error('*** sizes of "ref" and "refwgt" do not match')
  end
  [i j] = size(tstwgt);
  if i ~= sy
    error('*** sizes of "tst" and "tstwgt" do not match')
  end
end


%----- compute the loc distances ----- 

locdist = dist(ref(1:sx,:),tst(1:sy,:)');
slope = zeros(sx,sy)-1;
infin = 10^20;

locwgt_warp = eval(locweight_warp);  % local weights used for warping
locwgt_dist = eval(locweight_dist);  % local weights used for dist comp


%----- forward processing -----

dacc = ones(1,sy+2)*infin;                     % first column
dacc(3) = locdist(1,1)*locwgt_warp(1,1);

for ix = 2:sx,
  for iy = sy:-1:1,
    if (slope(ix-1,iy) == 1)%                  % if prev slope horizontal
      dacc(iy+2) = infin;
    end
    [mind,indx] = min(dacc(iy+2:-1:iy));
    slope(ix,iy) = indx*(mind<infin);
    dacc(iy+2) = mind+locdist(ix,iy)*locwgt_warp(ix,iy);
  end
end

%----- test if warping curve exists -----

if slope(sx,sy) == 0
  disp('warning: no warping curve found')
  dst = 999;
  warpcurve = [];
  distcurve = [];
  return;
end

%----- backward processing -----

warpcurve = zeros(sx,2);
distcurve = zeros(sx,1);
warpcurve(:,1) = (1:sx)';
warpcurve(sx,2) = sy;
ix = sx;
iy = sy;
distcurve(sx) = locdist(sx,sy)*locwgt_dist(sx,sy);
totweight = locwgt_dist(sx,sy);
iy = sy-slope(sx,sy)+1;
for ix = sx-1:-1:1,
  warpcurve(ix,2) = iy;
  distcurve(ix) = locdist(ix,iy)*locwgt_dist(ix,iy);
  totweight = totweight+locwgt_dist(ix,iy);
  iy = iy-slope(ix,iy)+1;
end
distcurve = distcurve/totweight*sx;
dst = sum(distcurve)/sx;

