function [p1,p2] = detect_endpoints(rms,dbg)
%
%  function [p1,p2] = detect_endpoints(rms,dbg)
%
%  Endpoints detection from intensity (RMS) contour; the maximum of the
%  histogram of the contour is used to determine the noise threshold.
%  If no utterance found, p1 and p2 are set to zero.
%
%  Inputs:    rms    RMS contour 
%             dbg    debug level; if dbg > 2, the RMS contour and the 
%                    RMS histogram are plotted in figure 4
%  
%  Outputs:   p1     detected utterance starts at RMS value number p1
%             p2     detected utterance ends at RMS value number p2
  
if max(rms) == 0
  p1 = [];
  p2 = [];
  if dbg > 0
    disp('warning:  empty signal')
  end
  return
end
if min(rms) < 0
  error('*** incorrect RMS contour:  intensity must not be negative!')
end

if nargin < 2
  dbg = 1;
end

rms = rms(:)';                    % make row vector
k = length(rms);                  % exclude trailer of zeros 
while rms(k) == 0
  k = k-1;
end

[n,x] = hist([0 rms(1:k)],40);    % determine noise/speech threshold
[y,m] = sort(n);
thresh = 3*x(m(length(n)));
if dbg > 2
  figure(4)
  clf
  h1 = axes('position',[0.1 0.6 0.8 0.3]);
  stem(rms,'r')
  set(gca,'xlim',[0 length(rms)+1])
  hold on
  plot([0 length(rms)+1],[1 1]*thresh,'-m','linewidth',1)
  title('RMS contour')
  axes('position',[0.1 0.1 0.8 0.3]);
  stairs([x-x(1) x(end)+x(1)],[n n(end)])
  set(gca,'xlim',[0 max(x(end)+x(1),thresh*1.2)])
  ylims = get(gca,'ylim');
  hold on
  plot([1 1]*thresh,ylims,'-m','linewidth',2)
  hold off
  text(thresh,0.7*ylims(2),'  Threshold','color','m')
  title('RMS histogram')
end

p1 = 1;	                         % determine starting point
while p1 < length(rms)-2    
  if (rms(p1) > thresh) & (rms(p1+2) > thresh)
    break
  end
  p1 = p1+1;
end  
p2 = length(rms);                % determine endpoint
while p2 > 3
  if (rms(p2) > thresh) & (rms(p2-2) > thresh)
    break
  end
  p2 = p2-1;
end

if p1 > p2 
  p1 = [];
  p2 = [];
  if dbg > 0
    disp('warning:  no endpoints detected')
  end
end

if dbg > 2 
  axes(h1);
  ylims = get(gca,'ylim');
  if ~isempty(p1)
    plot([p1 p1],ylims,':r')
  end
  if ~isempty(p2)
    plot([p2 p2],ylims,':g')
  end
  hold off
end
