function  [y,f] = mel_spectrum(x,fs,nfft,nfilt,comp,fig)
%
%  function  [y,f] = mel_spectrum(x,fs,nfft,nfilt,comp,fig)
%
%  Evaluate the mel spectrum of a signal segment 
%
%  Inputs:    x      signal segment
%             fs     sampling frequency
%             nfft   resolution of Fourier transform
%             nfilt  number of filters in mel filterbank
%             comp   if 1, compensate width of mel filters (not part 
%                    of ETSI standard but required for comparison 
%                    of mel and Fourier spectra)
%             fig    plot mel filter bank, Fourier spectrum and mel 
%                    spectrum in figure number `fig` (optional argument)
%
%  Outputs:   y      absolute value of mel spectrum (follow the ETSI 
%                    standard, the absolute value of the Fourier spectrum
%                    is filtert and thus `y` is real-valued too)
%             f      center frequencies [Hz] of mel filters (optional output)

x = x(:)';          % make row vector
if isempty(nfft)
  nfft = length(x);
end
if nfft < length(x)
  error(['*** resolution of FFT (' num2str(nfft) ...
         ') must not be smaller than window size ' num2str(length(x)) ')'])
end

X = fft(x,nfft);                       % Fourier spectrum
melfilt = melbankm(nfilt,nfft,fs)/2;   % mel filter bank
y = melfilt*abs(X(1:nfft/2+1)');       % mel spectrum

fs2_mel = 2595*log10(1+(fs/2)/700);
f_mel = (0:nfilt+1)*fs2_mel/(nfilt+1);  % uniform filters in mel domain
f_Hz = 700*(10.^(f_mel/2595)-1);        % mel filter center freqs in Hz

if comp == 1
  normfacs = (f_Hz(3:nfilt+2)-f_Hz(1:nfilt))/2/fs*nfft; 
  y = y./normfacs';
end

if nargin > 5 
  if isnumeric(fig) & fig > 0
    figure(fig)
    clf
    subplot(2,1,1);
    plot((0:nfft/2)/nfft*fs,melfilt');        % plot mel filter bank
    ylabel('gain')
    title(['mel filter bank   (' num2str(nfilt) ' channels)'])
  end
end

if nargout > 1
  f = f_Hz(2:nfilt+1);
end

