function s = logsweep(f0,f1,ph0,a,len) 
%
%  function:  s = logsweep(f0,f1,ph0,a,len)
%
%  Generates `len` samples of a logarithmically sweeped sine wave with 
%  amplitude `a`, and normalized start and end frequencies 'f0' and 'f1', 
%  resp. The start phase is `ph0` (in rad).

%  Note that in order to prevent numeric problems, this function changes 
%  for very small differences between 'f0' and 'f1' to a linear sweep! 

%  Frequency at time `t`: 
%          f(t) = f0*(f1/f0)^(t/T) 
%  
%  Phase at time `t`:
%                        (t)   
%          phi(t) = INTEGRAL{2*pi*f(t)} 
%                        (0)
%
%                 = 2*pi*f0*T/log(f1/f0) * ((f1/f0)^(t/T) - 1) + ph0

if (f0 < 0) | (f0 > 0.5)
  error('*** normalized start frequency out of range [0.0 ... 0.5]');
end
if (f1 < 0) | (f1 > 0.5)
  error('*** normalized end frequency out of range [0.0 ... 0.5]');
end

T = len;
t = 0:len-1;
k = f1/f0;
if abs(k-1) > 0.0001 
  phi = 2*pi*f0*T/log(k) *(k.^(t/T) - 1) + ph0;      % phase for log sweep
else
  phi = 2*pi* (f0*t + 0.5*(f1-f0).*t.*t/len) + ph0;  % phase for lin sweep
end
s = a.*sin(phi);   
